# coding: utf8
import time
import traceback
import subprocess


from multiprocessing import Process, Value, Manager, Lock


class Metrics(object):
    m = Manager()

    def __init__(self):
        self.active = Value('b', False)
        self.update_process = None

        self.lock = Lock()
        self.cpu_frequency = self.m.list()
        self.cpu_temperature = self.m.list()
        self.cpu_frequency_hgram = self.m.list()
        self.cpu_temperature_hgram = self.m.list()

    def get_cpu_frequency(self):
        frequencies = []

        try:
            output = subprocess.check_output(['grep', 'MHz', '/proc/cpuinfo'])
            for line in output.split('\n'):
                if not line:
                    continue
                parts = line.strip().split(' ')
                frequencies.append(float(parts[-1]))
        except Exception:
            trace = traceback.format_exc()
            print('-----get_cpu_frequency-----')
            print(trace)
            print('---------------------------')

        return frequencies

    def get_cpu_temperature(self):
        temperatures = []

        try:
            output = subprocess.check_output('sensors | grep Core', shell=True)
            for line in output.split('\n'):
                if not line:
                    continue
                parts = line.strip().split(' ')
                parts = [x for x in parts if x]

                # 2 logic cpu by 1 hard
                temperatures.append(float(parts[2]))
                temperatures.append(float(parts[2]))
        except Exception as e:
            trace = traceback.format_exc()
            print('-----get_cpu_temperature-----')
            print(trace)
            print('---------------------------')

        return temperatures

    def set_item(self, dest, index, value):
        while index > len(dest) - 1:
            dest.append((time.time(), 'None', 0.0))
        dest[index] = value

    def collect_numerical(self, data, dest, signal):
        if not data:
            return

        with self.lock:
            i = 0
            current_time = time.time()
            for i, val in enumerate(data):
                self.set_item(dest, i, (current_time, signal.format(i, ''), val))

            self.set_item(dest, i + 1, (current_time, signal.format('', '-min'), min(data)))
            self.set_item(dest, i + 2, (current_time, signal.format('', '-max'), max(data)))

    def create_hgram(self, step, max_val, values):
        hgram = {}
        i = 0
        while i <= max_val:
            hgram[i] = 0
            i += step

        for val in values:
            val = val // step * step
            if val not in hgram:
                hgram[val] = 0
            hgram[val] += 1

        return [[bound, hgram[bound]] for bound in sorted(hgram)]

    def collect_hgram(self, data, dest, signal, size, max_val):
        if not data:
            return

        with self.lock:
            current_time = time.time()
            hgram = self.create_hgram(size, max_val, data)
            self.set_item(dest, 0, (current_time, signal.format('', ''), hgram))

    def collect_cpu_frequency(self):
        self.collect_numerical(self.get_cpu_frequency(), self.cpu_frequency, 'cpu{}-frequency{}_attt')

    def collect_cpu_temperature(self):
        self.collect_numerical(self.get_cpu_temperature(), self.cpu_temperature, 'cpu{}-temperature{}_attt')

    def collect_cpu_frequency_hgram(self):
        self.collect_hgram(
            self.get_cpu_frequency(), self.cpu_frequency_hgram, 'cpu{}-frequency{}_ahhh', 100, 5000
        )

    def collect_cpu_temperature_hgram(self):
        self.collect_hgram(
            self.get_cpu_temperature(), self.cpu_temperature_hgram, 'cpu{}-temperature{}_ahhh', 10, 200
        )

    def data_collection(self):
        while self.active.value:
            self.collect_cpu_frequency()
            self.collect_cpu_temperature()
            self.collect_cpu_frequency_hgram()
            self.collect_cpu_temperature_hgram()
            time.sleep(2)

    def stats(self):
        with self.lock:
            all_stats = self.cpu_frequency[::]
            all_stats.extend(self.cpu_temperature)
            all_stats.extend(self.cpu_frequency_hgram)
            all_stats.extend(self.cpu_temperature_hgram)
        all_stats = sorted(all_stats, key=lambda x: x[0])
        return all_stats

    def start(self):
        if not self.active.value:
            self.update_process = Process(target=self.data_collection, args=())
            self.update_process.start()
            self.active.value = True
        return self.update_process

    def stop(self):
        if self.active.value:
            self.active.value = False
            time.sleep(3)
            if self.update_process.is_alive():
                self.update_process.terminate()
            self.update_process = None
