package apply

import (
	"log"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/hostctl/pkg/hostinfo"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/infra/hostctl/rpc"
)

type ApplyOpts struct {
	NoExecute bool
	Include   string
	File      string
	Noop      bool
	Verbose   bool
	Check     bool
}

func Apply() *cobra.Command {
	opts := &ApplyOpts{}
	cmd := &cobra.Command{
		Use:   "apply",
		Short: "Apply unit",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			if opts.File == "" {
				log.Fatalf("no file specified")
			}
			_, err := os.Stat(opts.File)
			if err != nil {
				log.Fatal(err)
			}
			// in check mode we run apply in no-execute mode and with empty HostInfo
			info := &pb.HostInfo{
				// we need to preset kernel version for check mode. With empty version we'll fall
				KernelRelease: "4.19.119-31.4",
			}
			if opts.Check {
				opts.NoExecute = true
			} else {
				var err error
				info, err = hostinfo.FromString(opts.Include)
				if err != nil {
					log.Fatal(err)
				}
			}
			jobEnv := pb.EnvType_REAL
			if opts.Noop {
				jobEnv = pb.EnvType_NOOP
			}
			h := &rpc.HostCtl{}
			_, err = h.Apply(&pb.ApplyRequest{
				Info:      info,
				UnitFile:  opts.File,
				Logfile:   "",
				Verbose:   opts.Verbose,
				JobEnv:    jobEnv,
				NoExecute: opts.NoExecute,
			})
			if err != nil {
				log.Fatal(err)
			}
		},
	}

	flags := cmd.Flags()
	flags.BoolVar(&opts.Noop, "noop", false, "no-op execution, dry run")
	flags.BoolVarP(&opts.NoExecute, "no-execute", "E", false, "pre-process only, do not execute. prints result in YAML.")
	flags.StringVarP(&opts.Include, "include", "I", opts.Include, "use host includes from file")
	flags.StringVarP(&opts.File, "file", "f", opts.File, "file to apply")
	flags.BoolVarP(&opts.Verbose, "verbose", "v", opts.Verbose, "verbose mode")
	flags.BoolVarP(&opts.Check, "check", "C", opts.Check, "Check mode. Validate spec without reading host include.")

	return cmd
}
