package cat

import (
	"fmt"
	"os"

	"a.yandex-team.ru/infra/hostctl/internal/term"
	"a.yandex-team.ru/infra/hostctl/internal/units"
	"a.yandex-team.ru/infra/hostctl/internal/units/storage"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	hostpb "a.yandex-team.ru/infra/hostctl/proto"
	"github.com/spf13/cobra"
	"google.golang.org/protobuf/proto"
)

func Cat() *cobra.Command {
	return &cobra.Command{
		Use:   "cat <name>",
		Short: "Show current slot spec",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			slotName := args[0]
			s, err := units.Slot(slotName, storage.NewReadonly(""))
			if err != nil {
				term.FatalF("Failed to load units: %s", err)
			}
			if s.Current() == nil {
				term.FatalF("No current revision in slot %s", slotName)
			}
			var spec proto.Message
			switch slotSpec := s.Current().Proto().Spec.(type) {
			case *hostpb.Rev_Pod:
				spec = slotSpec.Pod
			case *hostpb.Rev_PackageSet:
				spec = slotSpec.PackageSet
			case *hostpb.Rev_TimerJob:
				spec = slotSpec.TimerJob
			case *hostpb.Rev_SystemService:
				spec = slotSpec.SystemService
			case *hostpb.Rev_PortoDaemon:
				spec = slotSpec.PortoDaemon
			}
			buf, err := yamlutil.ProtoToYaml(spec)
			if err != nil {
				term.FatalF("Failed to render %s spec to yaml: %s", s.Current().Kind(), err)
			}
			_, _ = fmt.Fprintf(os.Stdout, "%s\n=======\n%s\n", s.Current().Kind(), buf)
		},
	}
}
