package list

import (
	"fmt"
	"io"
	"os"
	"sort"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/hostctl/internal/color"
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/term"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/infra/hostctl/rpc"
)

var (
	faintColor  = color.New(color.Faint)
	greenColor  = color.New(color.FgGreen)
	yellowColor = color.New(color.FgYellow)
	redColor    = color.New(color.FgRed)
	blueColor   = color.New(color.FgBlue)
)

type ListOpts struct {
	Format    string
	StateFile string
}

func List() *cobra.Command {
	opts := &ListOpts{
		Format:    "",
		StateFile: "",
	}
	cmd := &cobra.Command{
		Use:   "list",
		Short: "List units",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			h := &rpc.HostCtl{}
			resp, err := h.List(&pb.ListRequest{StateFile: opts.StateFile})
			if err != nil {
				term.FatalE(err)
			}
			switch opts.Format {
			case "1":
				fmt.Println(fmtNames(resp.State))
			case "":
				_ = fprintTree(os.Stdout, resp.State)
			case "json":
				fmt.Println(fmtListJSON(resp.State))
			default:
				term.FatalF("invalid format: %s", opts.Format)
			}
		},
	}
	cmd.Flags().StringVarP(&opts.Format,
		"output", "o",
		opts.Format, "output format: supported: ['1', 'json'], default: human readable")
	cmd.Flags().StringVarP(&opts.StateFile,
		"file", "f",
		opts.StateFile, "read from state file, default - builtin production path")
	return cmd
}

func fmtNames(state *pb.HostctlState) string {
	names := make([]string, 0, 30)
	for _, u := range state.Slots {
		names = append(names, u.Name)
	}
	sort.Strings(names)
	return strings.Join(names, "\n")
}

func fmtListJSON(state *pb.HostctlState) string {
	islots := make([]slot.Slot, len(state.Slots))
	for i, s := range state.Slots {
		islots[i] = slot.NewSlot(s)
	}
	l := &pb.List{
		Statuses: make([]*pb.SlotStatus, len(islots)),
	}
	for i, s := range islots {
		l.Statuses[i] = s.Status().Proto()
	}
	return term.FmtProto(l)
}

/* Helper view into slot protobuf providing getters to attributes. */
type slotView struct {
	proto *pb.Slot
}

func (sv *slotView) Kind() string {
	// Find current revision
	for _, rev := range sv.proto.Revs {
		if rev.Target == pb.RevisionTarget_CURRENT {
			return rev.Meta.Kind
		}
	}
	// No current, use first removed
	for _, rev := range sv.proto.Revs {
		return rev.Meta.Kind
	}
	// No revisions, but slot exists - return something
	return "Unknown"
}

func (sv *slotView) Name() string {
	return sv.proto.Name
}

func (sv *slotView) Version() string {
	// Find current revision
	for _, rev := range sv.proto.Revs {
		if rev.Target == pb.RevisionTarget_CURRENT {
			return rev.Meta.Version
		}
	}
	// No current, use first removed
	for _, rev := range sv.proto.Revs {
		return rev.Meta.Version
	}
	// No revisions, but slot exists - return something
	return "<unknown>"
}

func (sv *slotView) Ready() string {
	return sv.proto.Status.Ready.Status
}

func (sv *slotView) Pending() string {
	return sv.proto.Status.Pending.Status
}

func (sv *slotView) Changed() string {
	return sv.proto.Status.Changed.Status
}

/* Helper view into hostctl state, providing utilities to list slots. */
type view struct {
	proto *pb.HostctlState
}

func (v *view) units(kind string) []*slotView {
	rv := make([]*slotView, 0)
	// Here we assume that units are sorted in state,
	// thus avoid sorting slots again.
	for _, s := range v.proto.Slots {
		v := &slotView{s}
		if v.Kind() == kind {
			rv = append(rv, v)
		}
	}
	return rv
}

func (v *view) Timestamp() time.Time {
	return v.proto.UnitsTs.AsTime()
}

func (v *view) SystemServices() []*slotView {
	return v.units("SystemService")
}

func (v *view) PortoDaemons() []*slotView {
	return v.units("PortoDaemon")
}

func (v *view) TimerJobs() []*slotView {
	return v.units("TimerJob")
}

func (v *view) PackageSets() []*slotView {
	return v.units("PackageSet")
}

func (v *view) Pods() []*slotView {
	return v.units("HostPod")
}

func fReady(s string) string {
	if s == "True" {
		return greenColor.Sprint("•")
	} else {
		return redColor.Sprint("•")
	}
}

func fPending(p string) string {
	if p != "False" {
		return yellowColor.Sprint(" pending")
	}
	return ""
}

func fChanged(ch string) string {
	if ch != "False" {
		return blueColor.Sprint(" changed")
	}
	return ""
}

func fprintUnit(w io.Writer, sv *slotView) {
	_, _ = fmt.Fprintf(w, "  %s %s %s%s%s\n",
		fReady(sv.Ready()),
		sv.Name(),
		faintColor.Sprintf("(v.%s)", sv.Version()),
		fPending(sv.Pending()),
		fChanged(sv.Changed()),
	)
}

func fprintTree(w io.Writer, state *pb.HostctlState) error {
	v := view{state}
	_, _ = fmt.Fprintln(w, "System:")
	_, _ = fmt.Fprintf(w, "  Last-Updated: %s\n", v.Timestamp().Format("2006-01-02 15:04:05.000"))
	items := []struct {
		Name   string
		ListFn func() []*slotView
	}{
		{
			"PackageSets",
			v.PackageSets,
		},
		{
			"PortoDaemons",
			v.PortoDaemons,
		},
		{
			"SystemServices",
			v.SystemServices,
		},
		{
			"TimerJobs",
			v.TimerJobs,
		},
		{
			"HostPods",
			v.Pods,
		},
	}
	for _, item := range items {
		_, _ = fmt.Fprintf(w, "%s:\n", item.Name)
		for _, sv := range item.ListFn() {
			fprintUnit(w, sv)
		}
	}
	return nil
}
