package main

import (
	"log"
	"os"
	"runtime"

	"a.yandex-team.ru/infra/hostctl/cmd/apply"
	"a.yandex-team.ru/infra/hostctl/cmd/cat"
	"a.yandex-team.ru/infra/hostctl/cmd/list"
	"a.yandex-team.ru/infra/hostctl/cmd/manage"
	"a.yandex-team.ru/infra/hostctl/cmd/reinstall"
	"a.yandex-team.ru/infra/hostctl/cmd/remove"
	"a.yandex-team.ru/infra/hostctl/cmd/render"
	"a.yandex-team.ru/infra/hostctl/cmd/renderall"
	"a.yandex-team.ru/infra/hostctl/cmd/report"
	"a.yandex-team.ru/infra/hostctl/cmd/restart"
	"a.yandex-team.ru/infra/hostctl/cmd/show"
	"a.yandex-team.ru/infra/hostctl/cmd/state"
	"a.yandex-team.ru/infra/hostctl/cmd/status"
	"a.yandex-team.ru/infra/hostctl/cmd/version"
	"a.yandex-team.ru/infra/hostctl/internal/cmdutil"

	"github.com/spf13/cobra"
)

func setupProcess() {
	// Do not run too many threads:
	//   * we do not want to consume extra resources
	//   * we do not utilize extra parallelism
	//   * we strive for fast execution time
	//   extra threads - more bookkeeping, etc
	runtime.GOMAXPROCS(4)

	// Avoid locale-related output parsing problems
	err := os.Setenv("LC_ALL", "C")
	if err != nil {
		log.Fatalf("Cannot set LC_ALL=C: %v", err)
	}
}

/* setupTermLog sets flags to build golang logger disabling all extra fields,
   leaving only message itself. As this logger is used and intended for terminal messages
   only. */
func setupTermLog() {
	log.SetFlags(0)
}

func main() {
	setupProcess()
	setupTermLog()
	rootCmd := cmdutil.NewRoot("hostctl", "RTC host control")
	Register(rootCmd)
	err := (*cobra.Command)(rootCmd).Execute()
	if err != nil {
		log.Fatal(err)
	}
}

func Register(cmd *cmdutil.RootCmd) {
	cmd.RegisterExclusiveCommand(apply.Apply())
	cmd.RegisterExclusiveCommand(manage.Manage())
	cmd.RegisterExclusiveCommand(remove.Remove())
	cmd.RegisterCommand(list.List())
	cmd.RegisterCommand(show.Show())
	cmd.RegisterCommand(status.Status())
	cmd.RegisterExclusiveCommand(restart.Restart())
	cmd.RegisterCommand(render.Render())
	cmd.RegisterCommand(renderall.AllRender())
	cmd.RegisterExclusiveCommand(reinstall.Reinstall())
	cmd.RegisterExclusiveCommand(report.Report())
	cmd.RegisterCommand(version.Version())
	cmd.RegisterCommand(state.State())
	cmd.RegisterCommand(cat.Cat())
}
