package manage

import (
	"a.yandex-team.ru/infra/hostctl/internal/term"
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"runtime"
	"runtime/pprof"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/hostctl/internal/reporting"
	"a.yandex-team.ru/infra/hostctl/pkg/hostinfo"

	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/infra/hostctl/rpc"
)

const (
	manageTimeout = 15 * time.Minute
	varLib        = "/var/lib/ya-salt"
	repoBaseDir   = varLib + "/repo"
	repoCurrent   = repoBaseDir + "/current"
)

var watchdog = watchdogCtx{
	ctx:    context.Background(),
	fatalf: log.Fatalf,
}

type watchdogCtx struct {
	ctx    context.Context
	fatalf func(f string, arg ...interface{})
}

type ManageOpts struct {
	memProfile   string
	noOrly       bool
	noop         bool
	include      string
	overrideDirs []string
	verbose      bool
	orlyURL      string
	targetUnits  []string
	stdin        bool
	repoDir      string
	reportParams *reporting.ReportParams
}

func Manage() *cobra.Command {
	opts := &ManageOpts{
		memProfile:   "",
		noOrly:       false,
		noop:         false,
		include:      "",
		overrideDirs: make([]string, 0),
		verbose:      false,
		orlyURL:      "https://orly.yandex-team.ru",
		stdin:        false,
		reportParams: reporting.New(),
		repoDir:      repoCurrent,
	}
	cmd := &cobra.Command{
		Use:   "manage",
		Short: `Manage host units.`,
		Long: `Manage host units.

Main entry point:
  * reads current saved state
  * applies "incoming" updates from units definitions from '--units'
  * brings units to target state (install, restart, etc.)
  * reports to YASM/Juggler/HM-reporter

Units are read from:
  * /etc/hostman/porto-daemons.d - legacy directory (for backwards compatibility)
  * /var/lib/ya-salt/repo/current/*/units.d/ - remote dirs
  * /etc/hostman/units.d - manual persistent overrides
  * /run/hostman/units.d - manual runtime overrides

Unit dirs order matters - last file definition overrides previous ones. I.e
runtime overrides will always be preferred.`,
		Args: cobra.MaximumNArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			setupWatchdog(manageTimeout)
			info, err := hostinfo.FromString(opts.include)
			if err != nil {
				log.Fatal(err)
			}
			//load required params for reporting
			opts.reportParams.InferFromNoop(opts.noop)
			err = opts.reportParams.InferFromHostInfo(info)
			if err != nil {
				log.Fatal(err)
			}
			storageType := pb.StorageType_READWRITE
			jobEnv := pb.EnvType_REAL
			if opts.noop {
				storageType = pb.StorageType_READONLY
				jobEnv = pb.EnvType_NOOP
			}
			logFile, err := cmd.Flags().GetString("logfile")
			if err != nil {
				log.Fatalf("cannot get logfile flag: %v", err)
			}
			h := &rpc.HostCtl{}
			request := pb.ManageRequest{
				Info:    info,
				Logfile: logFile,
				Verbose: opts.verbose,
				Orly: &pb.OrlyConfig{
					OrlyUrl: opts.orlyURL,
					NoOrly:  opts.noOrly,
				},
				Storage:      storageType,
				JobEnv:       jobEnv,
				ReportParams: opts.reportParams.Proto(),
				RepoDir:      opts.repoDir,
				OverrideDirs: opts.overrideDirs,
			}
			if len(opts.targetUnits) > 0 && len(args) > 0 {
				term.FatalF("can not call --target-units with args at the same time")
			}
			if opts.stdin {
				y, err := ioutil.ReadAll(os.Stdin)
				if err != nil {
					log.Fatal(fmt.Errorf("cannot read unit from stdin: %w", err))
				}
				request.ManageMode = &pb.ManageRequest_Inline{Inline: &pb.ManageInline{Yaml: string(y)}}
			} else {
				if len(opts.targetUnits) > 0 {
					request.ManageMode = &pb.ManageRequest_Targets{Targets: &pb.ManageTargets{TargetUnits: opts.targetUnits}}
				} else if len(args) > 0 {
					request.ManageMode = &pb.ManageRequest_Targets{Targets: &pb.ManageTargets{TargetUnits: args}}
				} else {
					request.ManageMode = &pb.ManageRequest_All{All: &pb.ManageAllMode{}}
				}
			}
			opts.reportParams.InferFromManageRequest(&request)
			err = opts.reportParams.InferFromCmdline(cmd.Flags())
			if err != nil {
				log.Fatal(fmt.Errorf("cannot apply cmdline flags to reporting params: %w", err))
			}
			errs := opts.reportParams.Validate()
			if errs != nil {
				for _, err := range errs {
					log.Println(err)
				}
				log.Fatal(errors.New("reporting params validation failed"))
			}
			_, err = h.Manage(&request)
			if len(opts.memProfile) > 0 {
				f, err := os.Create(opts.memProfile)
				if err != nil {
					log.Println(fmt.Errorf("could not create memory profile: %s", err))
				}
				defer f.Close() // error handling omitted for example
				runtime.GC()    // get up-to-date statistics
				if err := pprof.WriteHeapProfile(f); err != nil {
					log.Println(fmt.Errorf("could not create memory profile: %s", err))
				}
			}
			if err != nil {
				log.Fatal(err)
			}
		},
	}

	flags := cmd.Flags()
	flags.StringVar(&opts.include,
		"include",
		opts.include,
		`host include info source:
* '<env>' to parse from HOST_INCLUDE env var
* '/some/path/to/host.json' to read from manually crafted json (not implemented)
* '<empty>' will read ya-salt spec and extract host includes from it`)
	flags.StringSliceVar(&opts.overrideDirs, "units", opts.overrideDirs,
		`override default unit directories, e.g /etc/hostman/porto-daemons.d/`)
	flags.BoolVar(&opts.noOrly, "no-orly", opts.noOrly, "if set will ignore orly policy")
	flags.StringVar(&opts.orlyURL, "orly-url", opts.orlyURL, "orly URL")
	flags.StringVar(&opts.memProfile, "mem-profile", opts.memProfile, "mem-profile file")
	flags.StringSliceVar(&opts.targetUnits, "target-units", opts.targetUnits,
		"units that we need to execute on this call (e.g juggler-agent-rtc,ebpf-agent)")
	flags.BoolVar(&opts.noop, "noop", opts.noop, "dry run, will not perform any changes")
	flags.BoolVarP(&opts.verbose, "verbose", "v", false, "verbose output")
	flags.BoolVar(&opts.stdin, "stdin", false, "read unit from stdin")
	flags.StringVar(&opts.repoDir, "repo", opts.repoDir, "current hostman repo dir")
	opts.reportParams.MustRegisterFlags(flags)

	return cmd
}

// Setup manage timeout using context deadline magic.
// Couldn't use more reliable method via alarm() and default SIGALRM handler (default - terminate).
// Golang runtime sets its sighandler for all signals and then routes signal to receivers via channels.
// signal.Reset(syscall.SIGALRM) just alters sigprocmask, but not resets sighandler for SIGALRM to SIG_DFL.
func setupWatchdog(timeout time.Duration) (context.Context, context.CancelFunc) {
	ctx, cancel := context.WithTimeout(watchdog.ctx, timeout)
	go func() {
		<-ctx.Done()
		if err := ctx.Err(); err != context.Canceled {
			watchdog.fatalf("Lockup/slow execution detected. Did not finish in %s. Exiting...", timeout)
		}
	}()
	return ctx, cancel
}
