package renderall

import (
	"bytes"
	"fmt"
	"html/template"
	"io/ioutil"
	"log"
	"os"
	"path"
	"sort"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/pkg/render"
	"a.yandex-team.ru/infra/hostctl/pkg/render/rtcutil"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"

	"github.com/spf13/cobra"
	"github.com/vbauerster/mpb"
	"github.com/vbauerster/mpb/decor"

	"a.yandex-team.ru/infra/hostctl/internal/color"
	"a.yandex-team.ru/infra/hostctl/internal/term"
)

type RenderAllOpts struct {
	Filename string
	Cluster  string
	Output   string
}

type renderResp struct {
	hosts  []string
	stage  string
	absent bool
	err    string
	pretty string
}

func AllRender() *cobra.Command {
	opts := &RenderAllOpts{}
	output := []string{"cli", "html"}
	cmd := &cobra.Command{
		Use:   "render-all <filename>",
		Short: "Render hostctl unit based on from host_info hm-reporter",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			opts.Filename = args[0]
			bar := initBar(opts.Cluster)
			renderer, err := rtcutil.NewClusterRendererWithParams(opts.Cluster, *rtcutil.NewClusterRendererParams().WithProgressTick(func(total int) {
				bar.SetTotal(int64(total), false)
				bar.Increment()
			}))
			if err != nil {
				log.Fatalf("Cannot create RTC cluster renderer: %v", err)
			}
			s, err := render.NewLocalRenderStorage(opts.Filename)
			if err != nil {
				log.Fatalf("Cannot create render storage: %v", err)
			}
			unitName := render.UnitNameFromPath(opts.Filename)
			resps, err := renderSpecOnLocationHosts(s, unitName, renderer)
			if err != nil {
				term.FatalE(err)
			}
			switch opts.Output {
			case "html":
				name := path.Base(args[0])
				html, err := fmtResultHTML(name, opts.Cluster, resps)
				if err != nil {
					term.FatalE(err)
				}
				fName := fmt.Sprintf("%s-%s.html", strings.ToLower(opts.Cluster), name)
				err = ioutil.WriteFile(fName, html, 0644)
				if err != nil {
					term.FatalE(err)
				}
				log.Printf("Saved result to %s...\n", fName)
			default:
				fmt.Printf("%s", fmtResultTerm(resps))
			}
		},
	}

	flags := cmd.Flags()
	flags.StringVarP(&opts.Cluster, "cluster", "c", "",
		fmt.Sprintf("HM reporter cluster: one of [%s]", strings.Join(rtcutil.GetClusterList(), ", ")))
	flags.StringVarP(&opts.Output, "output", "o", "cli",
		fmt.Sprintf("Output format one of [%s]", strings.Join(output, ", ")))
	return cmd
}

func renderSpecOnLocationHosts(s unitstorage.Storage, name string, r *rtcutil.ClusterRenderer) ([]*renderResp, error) {
	results, err := r.UnitFromStorage(s, name)
	if err != nil {
		return nil, err
	}
	rv := make([]*renderResp, 0)
	for _, result := range results {
		var pretty string
		var e string
		err := result.Error()
		if err == nil {
			pretty, err = result.Pretty()
			if err != nil {
				return nil, err
			}
			rv = append(rv, &renderResp{
				hosts:  result.Hosts(),
				stage:  string(slot.MetaFromProto(result.SlotMeta()).Stage()),
				absent: string(slot.MetaFromProto(result.SlotMeta()).Stage()) == "absent",
				pretty: pretty,
			})
		} else {
			// On rendering with error we can have empty result
			e = result.Error().Error()
			rv = append(rv, &renderResp{
				hosts: result.Hosts(),
				err:   e,
			})
		}
	}
	return rv, nil
}

func initBar(location string) *mpb.Bar {
	p := mpb.New(mpb.WithWidth(64), mpb.WithOutput(os.Stderr))
	barName := fmt.Sprintf("Rendering unit on '%s'", location)
	return p.AddBar(10000,
		mpb.PrependDecorators(
			decor.Name(barName, decor.WC{W: len(barName) + 1, C: decor.DidentRight}),
			decor.OnComplete(
				decor.AverageETA(decor.ET_STYLE_GO, decor.WC{W: 4}), "done",
			),
		),
		mpb.AppendDecorators(
			decor.CountersNoUnit("%d / %d", decor.WCSyncWidth),
		),
	)
}

func fmtResultTerm(res []*renderResp) string {
	b := strings.Builder{}
	for _, r := range res {
		sort.Strings(r.hosts)
		_, _ = color.New(color.FgGreen).Fprintf(&b, "Hosts (%d): %s\n", len(r.hosts), strings.Join(r.hosts, " "))
		b.WriteString(r.pretty)
		if r.err != "" {
			b.WriteString(color.RedString("ERR: "))
			b.WriteString(r.err)
		}
		b.WriteByte('\n')
	}
	return b.String()
}

var tmpl = `
{{$loc := .LocLow}}
<h1><a href="http://hm-{{ $loc }}.in.yandex.net/units/{{ .Name }}">[{{ .Location }}] {{ .Name }}</a></h1>
<div style="display:flex;">
{{range .Units }}
	<div>
		<h2> Stage: {{ .Stage }}<span style="opacity:0.3;">{{ .Removed }}</span></h2>
		<h2> unit </h2>
		<pre>{{ .Pretty }}</pre>
		<h2> hosts {{ .Count }}: </h2>
		{{if .Err }}
			<h2 style="color: red;"> err </h2>
			<p>{{ .Err }}</p>
		{{end}}
		<details><ul>
		{{range .Hosts}}
			<li><a href="http://hm-{{ $loc }}.in.yandex.net/hosts/{{ . }}">{{ . }}</a></li>
		{{end}}
		</ul></details>
	</div>
{{end}}
<div>
`

type TmplUnit struct {
	Pretty  string
	Stage   string
	Removed string
	Err     string
	Hosts   []string
	Count   int
}

type TmplData struct {
	Name     string
	Location string
	LocLow   string
	Units    []*TmplUnit
}

func fmtResultHTML(name, location string, res []*renderResp) ([]byte, error) {
	data := &TmplData{
		Name:     name,
		Location: location,
		LocLow:   strings.ToLower(location),
		Units:    make([]*TmplUnit, 0),
	}
	for _, resp := range res {
		removed := ""
		if resp.absent {
			removed = "(removed)"
		}
		data.Units = append(data.Units, &TmplUnit{
			Pretty:  resp.pretty,
			Hosts:   resp.hosts,
			Count:   len(resp.hosts),
			Stage:   resp.stage,
			Removed: removed,
			Err:     resp.err,
		})
	}
	t := template.New("t")
	t, err := t.Parse(tmpl)
	if err != nil {
		return nil, err
	}
	out := bytes.Buffer{}
	err = t.Execute(&out, data)
	if err != nil {
		return nil, err
	}
	return out.Bytes(), nil
}
