package report

import (
	"a.yandex-team.ru/infra/hostctl/internal/reporting"
	"a.yandex-team.ru/infra/hostctl/internal/term"
	"a.yandex-team.ru/infra/hostctl/pkg/hostinfo"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/infra/hostctl/rpc"

	"github.com/spf13/cobra"
)

type ReportOpts struct {
	include      string
	verbose      bool
	reportParams *reporting.ReportParams
}

func Report() *cobra.Command {
	opts := &ReportOpts{
		reportParams: reporting.New(),
	}
	cmd := &cobra.Command{
		Use:   "report",
		Short: "Report current hostctl state",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			info, err := hostinfo.FromString(opts.include)
			if err != nil {
				term.FatalE(err)
			}
			//load required params for reporting
			opts.reportParams.InferFromNoop(false)
			err = opts.reportParams.InferFromHostInfo(info)
			if err != nil {
				term.FatalE(err)
			}
			err = opts.reportParams.InferFromCmdline(cmd.Flags())
			if err != nil {
				term.FatalF("cannot apply cmdline flags to reporting params: %v", err)
			}
			errs := opts.reportParams.Validate()
			if errs != nil {
				for _, err := range errs {
					term.LogE(err)
				}
				term.FatalF("reporting params validation failed")
			}
			logFile, err := cmd.Flags().GetString("logfile")
			if err != nil {
				term.FatalF("cannot get logfile flag: %v", err)
			}
			h := &rpc.HostCtl{}
			_, err = h.Report(&pb.ReportRequest{
				Info:         info,
				Logfile:      logFile,
				Verbose:      opts.verbose,
				ReportParams: opts.reportParams.Proto(),
			})
			if err != nil {
				term.FatalE(err)
			}
		},
	}

	flags := cmd.Flags()
	flags.StringVar(&opts.include,
		"include",
		opts.include,
		`host include info source:
* '<env>' to parse from HOST_INCLUDE env var
* '/some/path/to/host.json' to read from manually crafted json (not implemented)
* '<empty>' will read ya-salt spec and extract host includes from it`)
	flags.BoolVarP(&opts.verbose, "verbose", "v", false, "verbose output")
	opts.reportParams.MustRegisterFlags(flags)
	return cmd
}
