package restart

import (
	"bytes"
	"fmt"

	"github.com/spf13/cobra"
	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/infra/hostctl/internal/term"
	"a.yandex-team.ru/infra/hostctl/internal/units"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/storage"
	"a.yandex-team.ru/library/go/core/log"
	azap "a.yandex-team.ru/library/go/core/log/zap"
)

type RestartOpts struct {
	UnitName string
	Verbose  bool
}

func Restart() *cobra.Command {
	opts := &RestartOpts{}
	cmd := &cobra.Command{
		Use:   "restart <unit-name>",
		Short: "Restart daemon for unit",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			opts.UnitName = args[0]
			logLevel := log.InfoLevel
			if opts.Verbose {
				logLevel = log.DebugLevel
			}
			lcfg := azap.ConsoleConfig(logLevel)
			l, err := azap.New(lcfg)
			if err != nil {
				term.FatalE(err)
			}
			unitEnv := env.Real(l, true, "", "")
			chl, err := units.Restart(unitEnv, opts.UnitName, storage.NewReadonly(""))
			if err != nil {
				term.FatalE(err)
			}
			for _, changeLog := range chl {
				if len(changeLog.Events) > 0 {
					b := &bytes.Buffer{}
					b.WriteString("\nChangelog:\n")
					n, _ := fmt.Fprint(b, "-----------------------\n")
					if err := yaml.NewEncoder(b).Encode(chl); err != nil {
						l.Errorf("Failed to marshal changelog: %v", err)
					}
					b.Write(bytes.Repeat([]byte("-"), n))
					l.Infof(b.String())
				}
				if err != nil {
					l.Errorf("Failed to execute rev job: %v", err)
				} else {
					l.Debug("Changelog: empty")
				}
			}
		},
	}

	flags := cmd.Flags()
	flags.BoolVarP(&opts.Verbose, "verbose", "v", opts.Verbose, "verbose mode")

	return cmd
}
