package show

import (
	"bytes"
	"fmt"

	"github.com/golang/protobuf/proto"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/term"
	"a.yandex-team.ru/infra/hostctl/internal/units"

	"a.yandex-team.ru/infra/hostctl/internal/units/storage"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
)

type ShowOpts struct {
	Format    string
	StateFile string
}

func Show() *cobra.Command {
	opts := &ShowOpts{
		Format: "",
	}
	cmd := &cobra.Command{
		Use:   "show <unit-name>",
		Short: "Show unit",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			name := args[0]
			s, err := units.Slot(name, storage.NewReadonly(opts.StateFile))
			if err != nil {
				term.FatalF("Failed to load units: %s.", err)
			}
			if s == nil {
				term.FatalF("Unit '%s' not found.", name)
			}
			var out string
			switch opts.Format {
			case "", "yaml":
				out = fmtSlot(s)
			case "json":
				out = fmtSlotJSON(s)
			default:
				term.FatalF("Invalid format '%s'.", opts.Format)
			}
			fmt.Println(out)
		},
	}
	showFlags := cmd.Flags()
	showFlags.StringVarP(&opts.Format,
		"output", "o",
		opts.Format, "output format: supported: ['yaml', 'json'], by default will print YAML")
	showFlags.StringVarP(&opts.StateFile,
		"file", "f", opts.StateFile,
		"read from state file, default - builtin production path")
	return cmd
}

func fmtSlot(slot slot.Slot) string {
	w := &bytes.Buffer{}
	w.Grow(4096) // Should be enough
	_, _ = fmt.Fprintf(w, "# === slot: %s ===\n", slot.Name())
	w.WriteString("---\n") // Split as YAML document
	// print revisions
	cur := slot.Current()
	if cur != nil {
		_, _ = fmt.Fprintf(w, "# current: %s@%.11s (kind=%s)\n", cur.Meta().Version, cur.ID(), slot.Kind())
		b, err := yamlutil.ProtoToYaml(cur.Proto())
		if err != nil {
			panic(err)
		}
		w.Write(b)
	}
	for _, r := range slot.Removed() {
		w.WriteString("---\n") // Split as YAML document
		_, _ = fmt.Fprintf(w, "# removed rev: %s@%.11s (kind=%s)\n", r.Meta().Version, r.ID(), slot.Kind())
		b, err := yamlutil.ProtoToYaml(r.Proto())
		if err != nil {
			panic(err)
		}
		w.Write(b)
	}
	return w.String()
}

func fmtSlotJSON(slot slot.Slot) string {
	return term.FmtProto(proto.MessageV2(slot.Proto()))
}
