package status

import (
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"fmt"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/color"
	"a.yandex-team.ru/infra/hostctl/internal/fmtutil"
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/term"
	"a.yandex-team.ru/infra/hostctl/internal/units"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/hostctl/internal/units/storage"
)

const circleGlyph = "●"

type StatusOpts struct {
	Format    string
	StateFile string
}

func Status() *cobra.Command {
	opts := &StatusOpts{
		Format: "short",
	}
	cmd := &cobra.Command{
		Use:   "status <name>",
		Short: "Show unit status",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			slotName := args[0]
			s, err := units.Slot(slotName, storage.NewReadonly(opts.StateFile))
			if err != nil {
				term.FatalF("Failed to load units: %s", err)
			}
			if s == nil {
				term.FatalF("Unit '%s' not found", slotName)
			}
			switch opts.Format {
			case "short":
				slotString := fmtStatusCLI(s, false)
				fmt.Printf("%s", slotString)
			case "long":
				slotString := fmtStatusCLI(s, true)
				fmt.Printf("%s", slotString)
			case "json":
				slotString := fmtStatusJSON(s)
				fmt.Println(slotString)
			default:
				term.FatalF("Invalid format: %s", opts.Format)
			}
		},
	}
	cmd.Flags().StringVarP(&opts.Format,
		"output", "o",
		opts.Format, "output format: supported: ['json, short, long'], "+
			"by default will print human in readable format")
	cmd.Flags().StringVarP(&opts.StateFile,
		"file", "f", opts.StateFile,
		"read from state file, default - builtin production path")
	return cmd
}

func fmtStatusCLI(slot slot.Slot, long bool) string {
	out := strings.Builder{}
	out.Grow(1024)
	status := slot.Status()
	var col color.Attribute
	if status.IsReady() {
		col = color.FgGreen
	} else {
		col = color.FgRed
	}
	cur := slot.Current()
	k := slot.Kind()
	_, _ = fmt.Fprintf(&out, "%s %s [%s",
		color.New(col).Sprint(circleGlyph),
		color.New(color.Bold).Sprint(slot.Name()),
		k)
	out.WriteString(fmtEnvMode(slot.Meta().EnvMode()))
	out.WriteString("]")
	_, _ = fmt.Fprintf(&out, " - %s\n", term.FmtVer(slot.Meta().Stage(), cur.Meta().Version, cur.ID()))
	if from := slot.Meta().Annotations["filename"]; from != "" {
		_, _ = fmt.Fprintf(&out, "%10s: %s\n", "Loaded", from)
	}
	// Format conditions
	out.WriteString(fmtutil.Condition("Ready", status.Ready))
	if k == kind.PortoDaemon || k == kind.SystemService {
		out.WriteString(fmtutil.Condition("Running", status.Running))
	}
	if long || !status.IsPending() {
		out.WriteString(fmtutil.Condition("Pending", status.Pending))
	}
	if long || status.IsChanged() {
		out.WriteString(fmtutil.Condition("Changed", status.Changed))
	}
	if long || status.IsThrottled() {
		out.WriteString(fmtutil.Condition("Throttled", status.Throttled))
	}
	if long || status.IsRemoved() {
		out.WriteString(fmtutil.Condition("Removed", status.Removed))
	}
	if long || status.IsConflicted() {
		out.WriteString(fmtutil.Condition("Conflicted", status.Conflicted))
	}
	if long || status.IsInstalled() {
		out.WriteString(fmtutil.Condition("Installed", status.Installed))
	}
	return out.String()
}

func fmtStatusJSON(slot slot.Slot) string {
	return term.FmtProto(slot.Status().Proto())
}

func fmtEnvMode(mode slot.EnvMode) string {
	if mode == slot.ModeReal {
		return ""
	}
	return fmt.Sprintf(" (%s)", mode)
}
