package edit

import (
	"errors"
	"fmt"
	"io"
	"io/fs"
	"io/ioutil"
	"os"
	"path"
	"strings"

	"a.yandex-team.ru/infra/hostctl/hmctl/location"
	"a.yandex-team.ru/infra/hostctl/hmctl/remote"
	"a.yandex-team.ru/infra/hostctl/hmctl/repo"
	"a.yandex-team.ru/infra/hostctl/hmctl/util"
	"a.yandex-team.ru/library/go/core/log"
	"github.com/spf13/cobra"
)

type Opts struct {
	Cluster string
	Repo    string
	Unit    string
}

const (
	units   = "units.d"
	daemons = "porto-daemons.d"
)

var lookupDirs = []string{units, daemons}

func (o *Opts) validate() error {
	if err := location.Clusters.Valid(location.Location(o.Cluster)); err != nil {
		return err
	}
	if err := remote.Remotes.Valid(remote.Remote(o.Repo)); err != nil {
		return err
	}
	return nil
}

func Edit() *cobra.Command {
	opts := &Opts{}
	cmd := &cobra.Command{
		Use:   "edit <unit-name>",
		Short: "Edit hostctl unit",
		Args:  cobra.ExactArgs(1),
		RunE: func(cmd *cobra.Command, args []string) error {
			if len(args) < 1 {
				return errors.New("no unit name provided")
			}
			if len(args) > 1 {
				return fmt.Errorf("multiple unit names provided: %v", args)
			}
			opts.Unit = args[0]
			err := opts.validate()
			if err != nil {
				return err
			}
			r, err := repo.Clone(remote.Remote(opts.Repo), location.Location(opts.Cluster))
			if err != nil {
				return err
			}
			var unitPath string
			for _, l := range lookupDirs {
				candidate := path.Join(l, fmt.Sprintf("%s.yaml", opts.Unit))
				_, err := r.Stat(candidate)
				if errors.Is(err, fs.ErrNotExist) {
					continue
				}
				if err != nil {
					return fmt.Errorf("failed to lookup candidate path %s: %w", candidate, err)
				}
				if unitPath != "" {
					return fmt.Errorf("unit exists in multiple directories from [%s]", strings.Join(lookupDirs, ", "))
				}
				unitPath = candidate
			}
			if unitPath == "" {
				return fmt.Errorf("unit %s not found", opts.Unit)
			}
			f, err := r.Open(unitPath)
			if err != nil {
				return err
			}
			tmp, err := ioutil.TempFile(".", opts.Unit)
			if err != nil {
				return err
			}
			defer os.Remove(tmp.Name())
			_, err = io.Copy(tmp, f)
			if err != nil {
				return fmt.Errorf("failed to copy to tmp file: %w", err)
			}
			err = tmp.Close()
			if err != nil {
				return err
			}
			err = f.Close()
			if err != nil {
				return err
			}
			editor := os.Getenv("EDITOR")
			if editor == "" {
				editor = "/usr/bin/vim"
			}
			err = util.RunEditor(editor, tmp.Name())
			if err != nil {
				return err
			}

			msg, err := ioutil.TempFile(".", fmt.Sprintf("commit-%s", opts.Unit))
			if err != nil {
				return fmt.Errorf("failed to open temp file for commit message")
			}
			defer os.Remove(msg.Name())
			_, err = msg.Write([]byte("Please enter your commit message\n"))
			if err != nil {
				return err
			}
			err = msg.Close()
			if err != nil {
				return err
			}
			err = util.RunEditor(editor, msg.Name())
			if err != nil {
				return err
			}
			msg, err = os.Open(msg.Name())
			if err != nil {
				return fmt.Errorf("failed to open edited commit message: %w", err)
			}
			defer msg.Close()
			commitMessage, err := io.ReadAll(msg)
			if err != nil {
				return err
			}
			logger := util.NewLogger(log.InfoLevel)
			pc := &util.PushConfig{
				Remote:         remote.Remote(opts.Repo),
				LocalFilePath:  tmp.Name(),
				Message:        string(commitMessage),
				RemoteFilePath: unitPath,
			}
			return util.PushRepo(logger, r, pc)
		},
	}
	flags := cmd.Flags()
	flags.StringVarP(&opts.Repo, "repo", "r", "", fmt.Sprintf("repo one of: %s", remote.Remotes.String()))
	flags.StringVarP(&opts.Cluster, "cluster", "c", "", fmt.Sprintf("cluster one of: %s", location.FmtClusters()))
	return cmd
}
