package infra

import (
	"a.yandex-team.ru/infra/hostctl/pkg/infra"
	"fmt"
	"github.com/spf13/cobra"
	"gopkg.in/yaml.v2"
	"io"
	"os"
	"time"
)

type Template struct {
	Name        string         `yaml:"name"`
	Description string         `yaml:"description"`
	Location    infra.Location `yaml:"location"`
	Duration    time.Duration  `yaml:"duration"`
}

func (t *Template) OverrideEvent(e *infra.DeploymentEvent) {
	e.Name = t.Name
	e.Location = t.Location
	e.Description = t.Description
	e.Duration = t.Duration
}

type Opts struct {
	Location infra.Location
	Duration time.Duration
	Percent  int
}

func (o *Opts) OverrideEvent(e *infra.DeploymentEvent) {
	if o.Location != "" {
		e.Location = o.Location
	}
	if o.Duration != 0 {
		e.Duration = o.Duration
	}
	e.Percent = o.Percent
}

func Infra() *cobra.Command {
	opts := &Opts{}
	cmd := &cobra.Command{
		Use:   "infra <template>",
		Short: "Create infra event by template. Cmdline flags override given template.",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			token := os.Getenv("INFRA_TOKEN")
			if token == "" {
				fmt.Println("Empty INFRA_TOKEN env. Receive token here: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=f824e92d0ff44e879084f2fe5c1b187b")
				return
			}
			c := infra.NewClient(token)
			e := &infra.DeploymentEvent{StartTime: time.Now()}
			t := &Template{}
			file, err := os.Open(args[0])
			if err != nil {
				fmt.Printf("Error: %s\n", err)
				return
			}
			templateContent, err := io.ReadAll(file)
			if err != nil {
				return
			}
			if err := yaml.Unmarshal(templateContent, t); err != nil {
				fmt.Printf("Error: %s\n", err)
				return
			}
			t.OverrideEvent(e)
			opts.OverrideEvent(e)
			if err := c.CreateDeploymentEvent(e); err != nil {
				fmt.Printf("Error: %s\n", err)
				return
			}
		},
	}
	flags := cmd.Flags()
	flags.StringVar((*string)(&opts.Location), "location", string(opts.Location),
		fmt.Sprintf("Location one of [%s, %s, %s, %s, %s]",
			infra.SasLocation,
			infra.VlaLocation,
			infra.ManLocation,
			infra.MskLocation,
			infra.AllLocation,
		),
	)
	flags.IntVar(&opts.Percent, "percent", opts.Percent, "Percent of partial deployment")
	flags.DurationVar(&opts.Duration, "duration", opts.Duration, "Event duration")
	return cmd
}
