package main

import (
	"errors"
	"fmt"
	"os"
	"path"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/infra/hostctl/hmctl/cmd/edit"
	"a.yandex-team.ru/infra/hostctl/hmctl/cmd/infra"
	"a.yandex-team.ru/infra/hostctl/hmctl/util"

	"a.yandex-team.ru/infra/hostctl/cmd/render"
	"a.yandex-team.ru/infra/hostctl/cmd/renderall"
	"a.yandex-team.ru/infra/hostctl/hmctl/cmd/killswitch"
	"a.yandex-team.ru/infra/hostctl/hmctl/location"
	"a.yandex-team.ru/infra/hostctl/hmctl/remote"
	"a.yandex-team.ru/library/go/core/log"
	"github.com/spf13/cobra"
)

type pushUnitConfig struct {
	*util.PushConfig
	Cluster location.Location
	Verbose bool
}

func (c *pushUnitConfig) prepare() error {
	name := filepath.Base(c.LocalFilePath)
	dir := filepath.Dir(c.LocalFilePath)
	if name == "" {
		return fmt.Errorf("empty file name in '%s'", c.LocalFilePath)
	}
	c.RemoteFilePath = path.Join("units.d", name)
	if c.Cluster != "" {
		return nil
	}
	// Try to parse cluster from name
	parts := strings.SplitN(name, "-", 2)
	if len(parts) == 0 {
		return fmt.Errorf("failed to deduce cluster from '%s'", name)
	}
	cluster := location.Location(parts[0])
	// Look if cluster is valid
	for _, cl := range location.Clusters {
		if string(cluster) == location.Branches[cl] {
			c.Cluster = cl
			break
		}
	}
	if c.Cluster == "" {
		return fmt.Errorf("invalid cluster in filename '%s'", name)
	}
	// Assemble file name back
	c.LocalFilePath = filepath.Join(dir, name)
	c.RemoteFilePath = parts[1]
	c.RemoteFilePath = path.Join("units.d/", c.RemoteFilePath)
	return nil
}

var rootCmd = &cobra.Command{
	Use:   "hmctl",
	Short: "Hostmanager repository utility",
}

func push(c *pushUnitConfig) error {
	l := util.NewLogger(log.InfoLevel)
	if c.Verbose {
		l = util.NewLogger(log.DebugLevel)
	}
	if c.Cluster == location.ALL {
		for _, loc := range location.Clusters {
			if err := util.PushLocation(l, c.PushConfig, loc); err != nil {
				return err
			}
		}
		return nil
	} else {
		return util.PushLocation(l, c.PushConfig, c.Cluster)
	}
}

func main() {
	pushConfig := &pushUnitConfig{PushConfig: &util.PushConfig{}}
	pushCmd := &cobra.Command{
		Use:   "push <unit yaml file>",
		Short: "Pushes specified unit to cluster",
		Long: `Push hostctl unit definition.
Accepts path to YAML file with hostctl unit definition and
pushes it to specified cluster triggering deployment.

If file name <cluster>-<filename-in-repo>.yaml, then --cluster flag can be omitted`,
		Args:         cobra.ExactArgs(1),
		SilenceUsage: true,
		RunE: func(cmd *cobra.Command, args []string) error {
			if len(args) < 1 {
				return errors.New("no file provided")
			}
			if len(args) > 1 {
				return fmt.Errorf("multiple files provided: %v", args)
			}
			pushConfig.LocalFilePath = args[0]
			if err := pushConfig.prepare(); err != nil {
				return nil
			}
			return push(pushConfig)
		},
	}
	rootCmd.AddCommand(pushCmd)
	rootCmd.AddCommand(render.Render())
	rootCmd.AddCommand(killswitch.Killswitch())
	rootCmd.AddCommand(renderall.AllRender())
	rootCmd.AddCommand(infra.Infra())
	rootCmd.AddCommand(edit.Edit())
	flags := pushCmd.Flags()
	flags.StringVarP((*string)(&pushConfig.Cluster), "cluster", "c", "",
		fmt.Sprintf("cluster to push to, one of: %s, '%s' to push everywhere", location.FmtClusters(), location.ALL))
	flags.StringVarP(&pushConfig.Message, "message", "m", "", "commit message")
	flags.StringVarP((*string)(&pushConfig.Remote), "remote", "r", string(remote.Test), fmt.Sprintf("repository remote, oneof: [%s]", remote.Remotes.String()))
	flags.BoolVarP(&pushConfig.Verbose, "verbose", "v", false, "")
	if err := rootCmd.Execute(); err != nil {
		os.Exit(1)
	}
	os.Exit(0)
}
