package remote

import (
	"fmt"
	"strings"
)

type Remote string
type remotes map[Remote]string

const (
	Core   Remote = "core"
	Sysdev Remote = "sysdev"
	Test   Remote = "test"
	Market Remote = "market"
	RND    Remote = "rnd"
	ISS    Remote = "iss"
	YASM   Remote = "yasm"

	TestURL   = "ssh://git@bb.yandex-team.ru/rtcsalt/test-hostman.git"
	CoreURL   = "ssh://git@bb.yandex-team.ru/rtcsalt/core.git"
	SysdevURL = "ssh://git@bb.yandex-team.ru/rtcsalt/sysdev.git"
	MarketURL = "ssh://git@bb.yandex-team.ru/rtcsalt/market.git"
	RNDURL    = "ssh://git@bb.yandex-team.ru/rtcsalt/rnd.git"
	ISSURL    = "ssh://git@bb.yandex-team.ru/rtcsalt/iss.git"
	YASMURL   = "ssh://git@bb.yandex-team.ru/rtcsalt/yasm.git"
)

var (
	Remotes = remotes{
		Core:   CoreURL,
		Sysdev: SysdevURL,
		Test:   TestURL,
		Market: MarketURL,
		RND:    RNDURL,
		ISS:    ISSURL,
		YASM:   YASMURL,
	}
)

func (r Remote) URL() (string, error) {
	if u, ok := Remotes[r]; ok {
		return u, nil
	}
	return "", fmt.Errorf("invalid remote '%s', should be one of [%s]", r, Remotes.String())
}

func (r remotes) String() string {
	strs := make([]string, 0, len(r))
	for remote := range r {
		strs = append(strs, string(remote))
	}
	return strings.Join(strs, ", ")
}

func (r remotes) Valid(rem Remote) error {
	if _, ok := Remotes[rem]; !ok {
		return fmt.Errorf("remote \"%s\" is invalid, valid remotes: [%s]", rem, r.String())
	}
	return nil
}
