package repo

import (
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"time"

	"github.com/go-git/go-billy/v5"
	"github.com/go-git/go-billy/v5/memfs"
	"github.com/go-git/go-git/v5"
	"github.com/go-git/go-git/v5/config"
	"github.com/go-git/go-git/v5/plumbing"
	"github.com/go-git/go-git/v5/plumbing/object"
	"github.com/go-git/go-git/v5/storage/memory"

	"a.yandex-team.ru/infra/hostctl/hmctl/location"
	"a.yandex-team.ru/infra/hostctl/hmctl/remote"
	"a.yandex-team.ru/library/go/core/log"
)

type Repo struct {
	repo         *git.Repository
	wt           *git.Worktree
	fs           billy.Filesystem
	s            *memory.Storage
	targetBranch plumbing.ReferenceName
	tmpBranch    plumbing.ReferenceName
	remote       remote.Remote
	unitPath     string
}

type File interface {
	io.Reader
	io.Closer
	io.Writer
	io.Seeker
}

func Clone(remote remote.Remote, loc location.Location) (*Repo, error) {
	branch, ok := location.Branches[loc]
	if !ok {
		return nil, fmt.Errorf("unknown cluster '%s' specified", loc)
	}
	url, err := remote.URL()
	if err != nil {
		return nil, err
	}
	targetBranch := plumbing.NewBranchReferenceName(branch)
	s := memory.NewStorage()
	fs := memfs.New()
	repo, err := git.Clone(s, fs, &git.CloneOptions{
		URL:           url,
		ReferenceName: targetBranch,
		SingleBranch:  true,
		Tags:          git.NoTags,
		Depth:         1,
	})
	if err != nil {
		return nil, err
	}
	wt, err := repo.Worktree()
	if err != nil {
		return nil, fmt.Errorf("work tree failed: %w", err)
	}
	return &Repo{
		repo:         repo,
		wt:           wt,
		fs:           fs,
		s:            s,
		targetBranch: targetBranch,
		remote:       remote,
	}, nil
}

func (r *Repo) CheckoutB(ref plumbing.ReferenceName) error {
	r.tmpBranch = ref
	return r.wt.Checkout(&git.CheckoutOptions{
		Branch: ref,
		Create: true,
		Keep:   true,
	})
}

func (r *Repo) Open(name string) (File, error) {
	return r.fs.Open(name)
}

func (r *Repo) Stat(name string) (os.FileInfo, error) {
	return r.fs.Stat(name)
}

func (r *Repo) CommitUnit(l log.Logger, message string, remotePath string, srcFilename string) (changed bool, err error) {
	r.unitPath = remotePath
	dst, err := r.wt.Filesystem.OpenFile(remotePath, os.O_RDWR|os.O_CREATE, 0644)
	if err != nil {
		return false, fmt.Errorf("failed to open %s in repo: %s", remotePath, err)
	}
	defer func() {
		_ = dst.Close()
	}()
	// Overwrite file content with new one.
	if err := dst.Truncate(0); err != nil {
		return false, err
	}
	src, err := os.Open(srcFilename)
	if err != nil {
		return false, err
	}
	defer func() {
		_ = src.Close()
	}()
	n, err := io.Copy(dst, src)
	if err != nil {
		return false, err
	}
	if n == 0 {
		return false, fmt.Errorf("provided file '%s' seems empty, aborting", srcFilename)
	}
	// Add file and commit
	_, err = r.wt.Add(remotePath)
	if err != nil {
		return false, err
	}
	status, err := r.wt.Status()
	if err != nil {
		return false, err
	}
	if status.IsClean() {
		return false, nil
	}
	sign, err := signature()
	if err != nil {
		return false, err
	}
	commit, err := r.wt.Commit(message, &git.CommitOptions{
		All:       false,
		Author:    sign,
		Committer: sign,
	})
	if err != nil {
		return false, err
	}
	return true, logCommit(l, r.repo, commit)
}

func signature() (*object.Signature, error) {
	c := config.NewConfig()
	home := os.Getenv("HOME")
	if home == "" {
		return nil, errors.New("no HOME env variable, cannot determine path to ~/.gitconfig")
	}
	buf, err := ioutil.ReadFile(home + "/" + ".gitconfig")
	if err != nil {
		return nil, err
	}
	if err := c.Unmarshal(buf); err != nil {
		return nil, err
	}
	u := c.Raw.Section("user")
	return &object.Signature{
		Name:  u.Option("name"),
		Email: u.Option("email"),
		When:  time.Now(),
	}, nil
}

func (r *Repo) Push() error {
	if err := r.repo.Push(&git.PushOptions{}); err != nil {
		return fmt.Errorf("push failed: %w", err)
	}
	return nil
}

func (r *Repo) PrURL() string {
	return fmt.Sprintf("https://bb.yandex-team.ru/projects/RTCSALT/repos/%s/pull-requests?create&sourceBranch=%s&targetBranch=%s",
		r.remote, r.tmpBranch.String(), r.targetBranch.String())
}

func logCommit(l log.Logger, repo *git.Repository, commit plumbing.Hash) error {
	l.Debug(commit.String())
	// Show diff
	obj, err := repo.CommitObject(commit)
	if err != nil {
		return fmt.Errorf("failed to get commit: %w", err)
	}
	for _, h := range obj.ParentHashes {
		l.Debug(h.String())
	}
	p, err := obj.Parent(0)
	if err != nil {
		return fmt.Errorf("failed to get parent commit: %w", err)
	}
	l.Debugf("%s", obj)
	patch, err := p.Patch(obj)
	if err != nil {
		l.Errorf("%s", err)
	} else {
		l.Debugf("%s", patch)
	}
	return nil
}
