package util

import (
	"fmt"
	"math/rand"
	"time"

	"a.yandex-team.ru/infra/hostctl/hmctl/location"
	"a.yandex-team.ru/infra/hostctl/hmctl/remote"
	"a.yandex-team.ru/infra/hostctl/hmctl/repo"
	"a.yandex-team.ru/library/go/core/log"
	"github.com/go-git/go-git/v5/plumbing"
)

type PushConfig struct {
	Remote         remote.Remote
	LocalFilePath  string
	Message        string
	RemoteFilePath string
}

func PushLocation(l log.Logger, c *PushConfig, loc location.Location) error {
	l.Infof("Pushing %s in %s location.", c.RemoteFilePath, loc)
	r, err := repo.Clone(c.Remote, loc)
	if err != nil {
		return err
	}
	return PushRepo(l, r, c)
}

func PushRepo(l log.Logger, r *repo.Repo, c *PushConfig) error {
	tmpBr := plumbing.NewBranchReferenceName(fmt.Sprintf("feature/%s-%s@%s", c.RemoteFilePath, time.Now().Format("2006.01.02"), randStr(6)))
	if err := r.CheckoutB(tmpBr); err != nil {
		return err
	}
	changed, err := r.CommitUnit(l, c.Message, c.RemoteFilePath, c.LocalFilePath)
	if err != nil {
		return err
	}
	if !changed {
		l.Warnf("Seems there is nothing to push: no changes.")
		return nil
	}
	if err := r.Push(); err != nil {
		return err
	}
	l.Infof("Done! Successfully pushed %s to %s.", c.RemoteFilePath, r.PrURL())
	l.Infof("Create Pull Request:\n%s", r.PrURL())
	return OpenBrowser(r.PrURL())
}

func randStr(n int) string {
	letterRunes := []rune("abcdefghijklmnopqrstuvwxyz")
	r := rand.New(rand.NewSource(time.Now().UnixNano()))
	b := make([]rune, n)
	for i := range b {
		b[i] = letterRunes[r.Intn(len(letterRunes))]
	}
	return string(b)
}
