package cmdutil

import (
	"fmt"
	"log"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/hostctl/internal/lockfile"
	"a.yandex-team.ru/infra/hostctl/rpc"
)

const defaultLog = "/var/log/hostctl.log"
const defaultHostctlLock = "/run/hostctl.lock"

type RootCmd cobra.Command
type RegisterFunc func(cmd *cobra.Command)

func NewRoot(use, long string) *RootCmd {
	root := &RootCmd{
		Use:              use,
		Long:             long,
		TraverseChildren: true,
	}
	(*cobra.Command)(root).PersistentFlags().String("logfile", defaultLog, "path to log file")
	(*cobra.Command)(root).PersistentFlags().String("lockfile", defaultHostctlLock, "path to lock file")
	err := (*cobra.Command)(root).PersistentFlags().MarkHidden("lockfile")
	if err != nil {
		panic(fmt.Errorf("cannot initialize lockfile flag: %w", err))
	}
	return root
}

func (root *RootCmd) RegisterCommand(cmd *cobra.Command) {
	(*cobra.Command)(root).AddCommand(cmd)
}

func (root *RootCmd) RegisterExclusiveCommand(cmd *cobra.Command) {
	wrappedFun := cmd.Run
	cmd.Run = func(cmd *cobra.Command, args []string) {
		path, err := cmd.Flags().GetString("lockfile")
		if err != nil {
			path = defaultHostctlLock
		}
		lock, err := lockfile.New(path)
		if err != nil {
			logFile, e := cmd.Flags().GetString("logfile")
			if e != nil {
				logFile = defaultLog
			}
			l, el := rpc.CreateLog(true, logFile)
			if el != nil {
				log.Fatal(el)
			}
			l.Errorf("%v", err)
			log.Fatal(err)
		}
		defer lock.Close()
		wrappedFun(cmd, args)
	}
	(*cobra.Command)(root).AddCommand(cmd)
}
