package hostctx

import (
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"github.com/stretchr/testify/assert"
	"testing"
)

func TestApplyCtx(t *testing.T) {
	meta := &pb.ObjectMeta{
		Kind:    "PackageSet",
		Name:    "name",
		Version: "{version}",
		Ctx: &pb.Context{
			Vars: []*pb.Context_Var{{
				Name: "version",
				Match: []*pb.Context_Match{{
					Exp: "geo('sas')",
					Val: "ver-1",
				}, {
					Exp: "default()",
					Val: "ver-2",
				}},
			}},
		},
	}
	spec := &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{{
			Name:    "pkg1",
			Version: "{version}",
		}},
		Files: []*pb.ManagedFile{{
			Path:    "/tmp/file1",
			Content: "content",
			User:    "vaspahomov",
			Group:   "vaspahomov",
			Mode:    "644",
		}},
	}
	ctxMap, err := Eval(testHostInfo, meta.Ctx)
	if err != nil {
		t.Fatal(err)
	}
	if err := ApplyCtx(meta, spec, ctxMap); err != nil {
		t.Fatal(err)
	}
	assert.Equal(t, "ver-1", meta.Version)
	assert.Equal(t, "ver-1", spec.Packages[0].Version)
}

func TestApplyCtxRemoveEmptyParamsFromCmdline(t *testing.T) {
	meta := &pb.ObjectMeta{
		Kind: "PortoDaemon",
		Name: "name",
		Ctx: &pb.Context{
			Vars: []*pb.Context_Var{{
				Name: "use-param",
				Match: []*pb.Context_Match{{
					Exp: "geo('sas')",
					Val: "",
				}, {
					Exp: "default()",
					Val: "--some-param",
				}},
			}},
		},
	}
	spec := &pb.PortoDaemon{
		Properties: &pb.PortoProperties{
			Cmd: []string{"python", ""},
		},
	}
	ctxMap, err := Eval(testHostInfo, meta.Ctx)
	if err != nil {
		t.Fatal(err)
	}
	if err := ApplyCtx(meta, spec, ctxMap); err != nil {
		t.Fatal(err)
	}
	assert.Equal(t, []string{"python"}, spec.Properties.Cmd)
}

func Test_InterpolatePb_notFailOnSystemService(t *testing.T) {
	spec := &pb.SystemServiceSpec{
		Packages:     make([]*pb.SystemPackage, 0),
		Files:        nil,
		UpdatePolicy: nil,
	}
	if err := interpolatePb(nil, spec); err != nil {
		t.Errorf("interpolatePb() error = %s", err)
	}
}

func Test_removeSkippedPackages(t *testing.T) {
	pkgs := []*pb.SystemPackage{{
		Name:    "p1",
		Version: "<skip>",
	}, {
		Name:    "p2",
		Version: "v1",
	}}
	expected := []*pb.SystemPackage{{
		Name:    "p2",
		Version: "v1",
	}}
	actual := filterSkippedPackages(pkgs)
	assert.Equal(t, expected, actual)
}
