package hostctx

import (
	"fmt"
	"regexp"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/engine/hostctx/celhelpers"

	"github.com/google/cel-go/cel"

	"a.yandex-team.ru/infra/hostctl/internal/engine/hostctx/funcs"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/slices"
)

var VarRe = regexp.MustCompile(`^[a-z][-a-z0-9]{1,60}$`)

const MaxValLen = 64 * 1024

func Eval(hi *pb.HostInfo, hostCtx *pb.Context) (map[string]string, error) {
	// Add "useful" string keys from include into result.
	// We could add all, but using lists for formatting does not seem practical.
	// Maybe someday we'll come upon with a better scheme.
	rv := map[string]string{
		"location":      hi.Location,
		"walle_project": hi.WalleProject,
		"dc":            hi.Dc,
		"dc_queue":      hi.DcQueue,
		"hostname":      hi.Hostname,
		"os_codename":   hi.OsCodename,
		"os_arch":       hi.OsArch,
	}
	// Put all includes into return value
	vars, err := initVars(hi)
	if err != nil {
		return nil, err
	}
	// Add empty vars values to add it to cel env.
	// We'll fill those values after calculating
	for _, v := range hostCtx.Vars {
		vars[v.Name] = ""
	}
	env, prgOpts, err := CreateEnv(hi, vars)
	if err != nil {
		return nil, err
	}
	errs := make([]error, 0)
	touchedVars := make([]string, 0)
	for i, v := range hostCtx.Vars {
		if v.Name == "" {
			errs = append(errs, fmt.Errorf("vars[%d].name is empty", i))
			continue
		}
		if !VarRe.MatchString(v.Name) {
			errs = append(errs, fmt.Errorf("vars[%d].name='%s' doesn't match '%s'", i, v.Name, VarRe.String()))
			continue
		}
		if slices.ContainsString(touchedVars, v.Name) {
			errs = append(errs, fmt.Errorf("vars[%d].name='%s' is duplicate", i, v.Name))
			continue
		}
		// Default values if nothing matches
		rv[v.Name] = ""
		vars[v.Name] = ""
		for _, m := range v.Match {
			ok, err := ExecExpr(env, prgOpts, m.Exp, vars)
			if err != nil {
				errs = append(errs, fmt.Errorf("failed to evaluate expr '%s' for var '%s': %v", m.Exp, v.Name, err))
			}
			if ok {
				if strings.ToLower(hostCtx.FormatValues) == "true" {
					// Allow using templates in context values
					rv[v.Name], err = fmtString(m.Val, rv)
					if err != nil {
						errs = append(errs, fmt.Errorf("failed to interpolate var '%s': %v", v.Name, err))
					}
				} else {
					rv[v.Name] = m.Val
				}
				if len(m.Val) > MaxValLen {
					errs = append(errs, fmt.Errorf("var '%s' is too long, max=%d", v.Name, MaxValLen))
				}
				vars[v.Name] = rv[v.Name]
				break
			}
		}
		touchedVars = append(touchedVars, v.Name)
	}
	if len(errs) > 0 {
		errStrings := make([]string, len(errs))
		for i := range errs {
			errStrings[i] = errs[i].Error()
		}
		return rv, fmt.Errorf("failed to validate and evaluate ctx:\n%s", strings.Join(errStrings, "\n"))
	}
	return rv, nil
}

func CreateEnv(hi *pb.HostInfo, vars map[string]interface{}) (*cel.Env, []cel.ProgramOption, error) {
	opts, fs := funcs.EnvOptions(hi)
	opts = append(opts, funcs.VarsDecls(vars)...)
	env, err := cel.NewEnv(cel.Declarations(opts...))
	if err != nil {
		return nil, nil, err
	}
	return env, []cel.ProgramOption{cel.Functions(fs...)}, nil
}

func ExecExpr(env *cel.Env, prgOpts []cel.ProgramOption, expr string, contextVars map[string]interface{}) (bool, error) {
	ast, iss := env.Compile(fixPython(expr))
	// iss.Err() expensive call
	// iss.Err() called in env.Compile()
	// and when iss.Err() not nil func returns nil ast
	if ast == nil {
		if err := iss.Err(); err != nil {
			return false, err
		}
	}
	prg, err := env.Program(ast, prgOpts...)
	if err != nil {
		return false, err
	}
	out, _, err := prg.Eval(contextVars)
	if err != nil {
		return false, err
	}
	return out.Value().(bool), nil
}

func initVars(host *pb.HostInfo) (map[string]interface{}, error) {
	k, err := celhelpers.ParseKernelRelease(host.KernelRelease)
	if err != nil {
		return nil, err
	}
	return map[string]interface{}{
		"hostname":      host.Hostname,
		"num":           host.Num,
		"walle_project": host.WalleProject,
		"walle_tags":    host.WalleTags,
		"net_switch":    host.NetSwitch,
		"gencfg_groups": host.GencfgGroups,
		"location":      host.Location,
		"dc":            host.Dc,
		"dc_queue":      host.DcQueue,
		"kernel":        k.ToMap(),
		"cpu_model":     host.CpuModel,
		"mem_total_mib": host.MemTotalMib,
		"os_codename":   host.OsCodename,
		"os_arch":       host.OsArch,
	}, nil
}

// hack to replace 'and' -> '&&' and 'or' -> '||'
// for python PortoDaemon backward compatibility
func fixPython(expr string) string {
	expr = strings.Replace(expr, " and ", " && ", -1)
	expr = strings.Replace(expr, " or ", " || ", -1)
	return expr
}
