package hostctx

import (
	"fmt"
	"strings"
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/engine/hostctx/celhelpers"

	"github.com/stretchr/testify/assert"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

var testHostInfo = &pb.HostInfo{
	Hostname:      "sas1-0000.search.yandex.net",
	Num:           17,
	WalleProject:  "rtc",
	WalleTags:     []string{"rtc.stage-production", "rtc.stage-experiment", "report_parallel"},
	NetSwitch:     "sas1-s000",
	GencfgGroups:  []string{"GCFG_GROUP1", "GCFG_GROUP2"},
	Location:      "sas",
	Dc:            "SASTA",
	DcQueue:       "sas1.2.1",
	KernelRelease: "4.19.119-30.2",
	MemTotalMib:   64383,
	OsCodename:    "xenial",
	OsArch:        "amd64",
}
var testFocalHostInfo = &pb.HostInfo{
	Hostname:      "sas1-0000.search.yandex.net",
	Num:           17,
	WalleProject:  "rtc",
	WalleTags:     []string{"rtc.stage-production", "rtc.stage-experiment", "report_parallel"},
	NetSwitch:     "sas1-s000",
	GencfgGroups:  []string{"GCFG_GROUP1", "GCFG_GROUP2"},
	Location:      "sas",
	Dc:            "SASTA",
	DcQueue:       "sas1.2.1",
	KernelRelease: "4.19.119-30.2",
	MemTotalMib:   64383,
	OsCodename:    "focal",
	OsArch:        "amd64",
}
var testFocalArm64HostInfo = &pb.HostInfo{
	Hostname:      "sas1-0000.search.yandex.net",
	Num:           17,
	WalleProject:  "rtc",
	WalleTags:     []string{"rtc.stage-production", "rtc.stage-experiment", "report_parallel"},
	NetSwitch:     "sas1-s000",
	GencfgGroups:  []string{"GCFG_GROUP1", "GCFG_GROUP2"},
	Location:      "sas",
	Dc:            "SASTA",
	DcQueue:       "sas1.2.1",
	KernelRelease: "4.19.119-30.2",
	MemTotalMib:   64383,
	OsCodename:    "focal",
	OsArch:        "arm64",
}

func TestEval_Validation(t *testing.T) {
	badCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "1has_tag('rtc.stage-production')",
				Val: "1",
			}, {
				Exp: "1default()",
				Val: "5",
			}},
		}},
	}
	_, err := Eval(testHostInfo, badCtx)
	if err == nil {
		t.Fatalf("Unexpected nil error")
	}
	expected := fmt.Errorf("failed to validate and evaluate ctx:\n%s", strings.Join([]string{
		`failed to evaluate expr '1has_tag('rtc.stage-production')' for var 'version': ERROR: <input>:1:2: Syntax error: mismatched input 'has_tag' expecting <EOF>
 | 1has_tag('rtc.stage-production')
 | .^`,
		`failed to evaluate expr '1default()' for var 'version': ERROR: <input>:1:2: Syntax error: mismatched input 'default' expecting <EOF>
 | 1default()
 | .^`,
	}, "\n"))
	assert.Equal(t, expected, err)
	// Test good case
	okCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "has_tag('rtc.stage-production')",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	_, err = Eval(testHostInfo, okCtx)
	if err != nil {
		t.Fatalf("Unexpected error: %s", err)
	}
}

func TestEvalCtx_HasTag(t *testing.T) {
	hostCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "has_tag('rtc.stage-production')",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	actual, err := Eval(testHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	// Check that host include strings are present in return value
	expected := map[string]string{
		"dc":            "SASTA",
		"dc_queue":      "sas1.2.1",
		"location":      "sas",
		"version":       "1",
		"walle_project": "rtc",
		"hostname":      "sas1-0000.search.yandex.net",
		"os_codename":   "xenial",
		"os_arch":       "amd64",
	}
	assert.Equal(t, expected, actual)
}

func TestEvalCtx_Num(t *testing.T) {
	hostCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "num <= 100",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	actual, err := Eval(testHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	// Check that host include strings are present in return value
	expected := map[string]string{
		"dc":            "SASTA",
		"dc_queue":      "sas1.2.1",
		"location":      "sas",
		"version":       "1",
		"walle_project": "rtc",
		"hostname":      "sas1-0000.search.yandex.net",
		"os_codename":   "xenial",
		"os_arch":       "amd64",
	}
	assert.Equal(t, expected, actual)
}
func TestEvalCtx_Prj(t *testing.T) {
	hostCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "walle_project == 'rtc'",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	actual, err := Eval(testHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	// Check that host include strings are present in return value
	expected := map[string]string{
		"dc":            "SASTA",
		"dc_queue":      "sas1.2.1",
		"location":      "sas",
		"version":       "1",
		"walle_project": "rtc",
		"hostname":      "sas1-0000.search.yandex.net",
		"os_codename":   "xenial",
		"os_arch":       "amd64",
	}
	assert.Equal(t, expected, actual)
}

func TestEvalCtx_DCQueue(t *testing.T) {
	hostCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "dc_queue == 'sas1.2.1'",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	actual, err := Eval(testHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	expected := map[string]string{
		"dc":            "SASTA",
		"dc_queue":      "sas1.2.1",
		"location":      "sas",
		"version":       "1",
		"walle_project": "rtc",
		"hostname":      "sas1-0000.search.yandex.net",
		"os_codename":   "xenial",
		"os_arch":       "amd64",
	}
	assert.Equal(t, expected, actual)
}

func TestEvalCtx_DCQueueFunc(t *testing.T) {
	hostCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "dc_queue('sas1.2.1')",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	actual, err := Eval(testHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	expected := map[string]string{
		"dc":            "SASTA",
		"dc_queue":      "sas1.2.1",
		"location":      "sas",
		"version":       "1",
		"walle_project": "rtc",
		"hostname":      "sas1-0000.search.yandex.net",
		"os_codename":   "xenial",
		"os_arch":       "amd64",
	}
	assert.Equal(t, expected, actual)
}

func TestEvalCtx_DCQueueListFunc(t *testing.T) {
	hostCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "dc_queue(['sas2.2.1', 'sas1.2.1'])",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	actual, err := Eval(testHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	expected := map[string]string{
		"dc":            "SASTA",
		"dc_queue":      "sas1.2.1",
		"location":      "sas",
		"version":       "1",
		"walle_project": "rtc",
		"hostname":      "sas1-0000.search.yandex.net",
		"os_codename":   "xenial",
		"os_arch":       "amd64",
	}
	assert.Equal(t, expected, actual)
}

func TestExecExpr_Production(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "production()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestExecExpr_PythonOr(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "false or false", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "true or false", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "true or true", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "false or true", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestExecExpr_PythonAnd(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "false and false", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "true and false", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "true and true", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "false and true", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
}

func TestExecExpr_Negation(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "!has_tag('qloud')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestExecExpr_Prestable(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "prestable()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
}

func TestExecExpr_Focal(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "focal()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	env, prgOpts, err = CreateEnv(testFocalHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err = ExecExpr(env, prgOpts, "focal()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestExecExpr_Xenial(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "xenial()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	env, prgOpts, err = CreateEnv(testFocalHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err = ExecExpr(env, prgOpts, "xenial()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
}

func TestExecExpr_Perc(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "perc(10)", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "perc(20)", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestExecExpr_HasTag(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "has_tag('rtc.stage-experiment')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "has_tag('rtc.stage-prestable')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "has_tag(['rtc.stage-experiment', 'rtc.stage-prestable'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "has_tag(['rtc.stage-prestable', 'rtc.some-tag'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	// No overload for has_tag() without arguments
	ok, err = ExecExpr(env, prgOpts, "has_tag()", make(map[string]interface{}))
	assert.False(t, ok)
	assert.Error(t, err)
}

func TestExecExpr_HasGroup(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, map[string]interface{}{"gencfg_groups": make([]string, 0)})
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "has_group('GCFG_GROUP1')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "'GCFG_GROUP1' in gencfg_groups", map[string]interface{}{"gencfg_groups": []string{"GCFG_GROUP1", "GCFG_GROUP2"}})
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "has_group('GCFG_GROUP3')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "'GCFG_GROUP3' in gencfg_groups", map[string]interface{}{"gencfg_groups": []string{"GCFG_GROUP1", "GCFG_GROUP2"}})
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "has_group(['GCFG_GROUP1', 'GCFG_GROUP3'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "has_group(['GCFG_GROUP3', 'GCFG_GROUP4'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	// No overload for has_group() without arguments
	ok, err = ExecExpr(env, prgOpts, "has_group()", make(map[string]interface{}))
	assert.False(t, ok)
	assert.Error(t, err)
}

func TestExecExpr_Geo(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "geo('sas')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "geo('man')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "geo(['sas', 'man'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "geo(['man', 'vla'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	// No overload for geo() without arguments
	ok, err = ExecExpr(env, prgOpts, "geo()", make(map[string]interface{}))
	assert.False(t, ok)
	assert.Error(t, err)
}

func TestExecExpr_Project(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "prj('rtc')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "prj(['rtc'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "prj(['rtc', 'rtc-mtn'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "prj(['rtc-mtn', 'rtc-yabs'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "prj('rtc-mtn')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	// No overload for prj() without arguments
	ok, err = ExecExpr(env, prgOpts, "prj()", make(map[string]interface{}))
	assert.False(t, ok)
	assert.Error(t, err)
}

func TestExecExpr_Host(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "h('sas1-0000')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "h('sas1-0000.search.yandex.net')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "h('sas1-0001.search.yandex.net')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "h('sas1-0001')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "h(['sas1-0000', 'sas1-0001'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "h(['sas1-0000.search.yandex.net', 'sas1-0001.search.yandex.net'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "h(['sas1-0001.search.yandex.net', 'sas1-0002.search.yandex.net'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	// No overload for h() without arguments
	ok, err = ExecExpr(env, prgOpts, "h()", make(map[string]interface{}))
	assert.False(t, ok)
	assert.Error(t, err)
}

func TestExecExpr_Switch(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "switch('sas1-s000')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "switch('sas1-s001')", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "switch(['sas1-s000', 'sas1-s001'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "switch(['sas1-s002', 'sas1-s001'])", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	// No overload for switch() without arguments
	ok, err = ExecExpr(env, prgOpts, "switch()", make(map[string]interface{}))
	assert.False(t, ok)
	assert.Error(t, err)
}

func TestExecExpr_Kernel(t *testing.T) {
	ker, err := celhelpers.ParseKernelRelease("4.19.119-30.2")
	if err != nil {
		t.Error(err)
	}
	vars := map[string]interface{}{"kernel": ker.ToMap()}
	env, prgOpts, err := CreateEnv(testHostInfo, vars)
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "kernel.major >= 4", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "kernel.major >= 5", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "kernel.minor >= 19", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "kernel.minor == 4", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "kernel.build == -1", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "kernel.patch == 119", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "kernel.patch < 119", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
	ok, err = ExecExpr(env, prgOpts, "kernel.release == '4.19.119-30.2'", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
	ok, err = ExecExpr(env, prgOpts, "kernel.release == '4.19.119-30.1'", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.False(t, ok)
}

func TestExecExpr_Default(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "default()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestExecExpr_Invalid(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "some+ aaa {} perl", make(map[string]interface{}))
	assert.False(t, ok)
	assert.Error(t, err)
}

func TestExecMemTotalMibAvailable(t *testing.T) {
	vars, err := initVars(testHostInfo)
	if err != nil {
		t.Fatal(err)
	}
	env, prgOpts, err := CreateEnv(testHostInfo, vars)
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "mem_total_mib > 1024", vars)
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestEvalCtx_FormatValues(t *testing.T) {
	hostCtx := &pb.Context{
		FormatValues: "True",
		Vars: []*pb.Context_Var{{
			Name: "shm-size",
			Match: []*pb.Context_Match{{
				Exp: "mem_total_mib < 120000",
				Val: "32G",
			}, {
				Exp: "default()",
				Val: "328G",
			}},
		},
			{
				Name: "shm-stage",
				Match: []*pb.Context_Match{{
					Exp: "has_tag('report_parallel')",
					Val: "-{shm-size}",
				}, {
					Exp: "default()",
					Val: "",
				}},
			}},
	}
	actual, err := Eval(testHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	// Check that host include strings are present in return value
	assert.Equal(t, "-32G", actual["shm-stage"])
}

func TestEvalCtx_NotFormatValuesDefault(t *testing.T) {
	hostCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "shm-size",
			Match: []*pb.Context_Match{{
				Exp: "mem_total_mib < 120000",
				Val: "32G",
			}, {
				Exp: "default()",
				Val: "328G",
			}},
		},
			{
				Name: "shm-stage",
				Match: []*pb.Context_Match{{
					Exp: "has_tag('report_parallel')",
					Val: "-{shm-size}",
				}, {
					Exp: "default()",
					Val: "",
				}},
			}},
	}
	actual, err := Eval(testHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	// Check that host include strings are present in return value
	assert.Equal(t, "-{shm-size}", actual["shm-stage"])
}

func TestExecExpr_AMD64(t *testing.T) {
	env, prgOpts, err := CreateEnv(testHostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "amd64()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestExecExpr_ARM64(t *testing.T) {
	env, prgOpts, err := CreateEnv(testFocalArm64HostInfo, make(map[string]interface{}))
	if err != nil {
		t.Error(err)
	}
	ok, err := ExecExpr(env, prgOpts, "arm64()", make(map[string]interface{}))
	if err != nil {
		t.Fatal(err)
	}
	assert.True(t, ok)
}

func TestVar_OSArch(t *testing.T) {
	hostCtx := &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "os_arch == 'arm64'",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	actual, err := Eval(testFocalArm64HostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	expected := map[string]string{
		"dc":            "SASTA",
		"dc_queue":      "sas1.2.1",
		"location":      "sas",
		"version":       "1",
		"walle_project": "rtc",
		"hostname":      "sas1-0000.search.yandex.net",
		"os_codename":   "focal",
		"os_arch":       "arm64",
	}
	assert.Equal(t, expected, actual)

	hostCtx = &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "version",
			Match: []*pb.Context_Match{{
				Exp: "os_arch == 'amd64'",
				Val: "1",
			}, {
				Exp: "default()",
				Val: "5",
			}},
		}},
	}
	actual, err = Eval(testFocalHostInfo, hostCtx)
	if err != nil {
		t.Fatal(err)
	}
	expected = map[string]string{
		"dc":            "SASTA",
		"dc_queue":      "sas1.2.1",
		"location":      "sas",
		"version":       "1",
		"walle_project": "rtc",
		"hostname":      "sas1-0000.search.yandex.net",
		"os_codename":   "focal",
		"os_arch":       "amd64",
	}
	assert.Equal(t, expected, actual)
}
