package funcs

import (
	"reflect"
	"strings"

	"github.com/google/cel-go/checker/decls"
	"github.com/google/cel-go/common/types"
	"github.com/google/cel-go/common/types/ref"
	"github.com/google/cel-go/interpreter/functions"
	exprpb "google.golang.org/genproto/googleapis/api/expr/v1alpha1"

	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/slices"
)

func EnvOptions(host *pb.HostInfo) (decls []*exprpb.Decl, funcs []*functions.Overload) {
	funcs, funcsDecls := make([]*functions.Overload, 0), make([]*exprpb.Decl, 0)
	funcs, funcsDecls = HasTagFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = HasGroupFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = GeoFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = PrestableFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = ProductionFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = ProjectFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = DCQueueFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = PercFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = HostFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = SwitchFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = DefaultFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = FocalFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = XenialFunc(host, funcs, funcsDecls)
	funcs, funcsDecls = AMD64Func(host, funcs, funcsDecls)
	funcs, funcsDecls = ARM64Func(host, funcs, funcsDecls)
	return funcsDecls, funcs
}

func VarsDecls(vars map[string]interface{}) []*exprpb.Decl {
	varsDecls := make([]*exprpb.Decl, 0)
	for name, val := range vars {
		switch reflect.TypeOf(val).Kind() {
		case reflect.String:
			varsDecls = append(varsDecls, decls.NewVar(name, decls.String))
		case reflect.Slice:
			varsDecls = append(varsDecls, decls.NewVar(name, decls.NewListType(decls.String)))
		case reflect.Int:
			varsDecls = append(varsDecls, decls.NewVar(name, decls.Int))
		case reflect.Int32:
			varsDecls = append(varsDecls, decls.NewVar(name, decls.Int))
		case reflect.Int64:
			varsDecls = append(varsDecls, decls.NewVar(name, decls.Int))
		case reflect.Float64:
			varsDecls = append(varsDecls, decls.NewVar(name, decls.Double))
		default:
			varsDecls = append(varsDecls, decls.NewVar(name, decls.Dyn))
		}
	}
	return varsDecls
}

func HasTagFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	hasTag := func(tag string, walleTags []string) bool {
		return slices.ContainsString(walleTags, tag)
	}
	f, d := &functions.Overload{
		Operator: "has_tag_string",
		Unary: func(lhs ref.Val) ref.Val {
			val := lhs.Value().(string)
			return types.Bool(hasTag(val, host.WalleTags))
		},
	}, decls.NewFunction("has_tag",
		decls.NewOverload("has_tag_string",
			[]*exprpb.Type{decls.String},
			decls.Bool))
	fList, dList := &functions.Overload{
		Operator: "has_tag_string_list",
		Unary: func(values ref.Val) ref.Val {
			list := values.Value().([]ref.Val)
			for _, tag := range list {
				t := tag.Value().(string)
				if hasTag(t, host.WalleTags) {
					return types.Bool(true)
				}
			}
			return types.Bool(false)
		},
	}, decls.NewFunction("has_tag",
		decls.NewOverload("has_tag_string_list",
			[]*exprpb.Type{decls.NewListType(decls.String)},
			decls.Bool))
	return append(funcs, f, fList), append(declarations, d, dList)
}

func HasGroupFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	hasGroup := func(group string, groups []string) bool {
		return slices.ContainsString(groups, group)
	}
	f, d := &functions.Overload{
		Operator: "has_group_string",
		Unary: func(lhs ref.Val) ref.Val {
			val := lhs.Value().(string)
			return types.Bool(hasGroup(val, host.GencfgGroups))
		},
	}, decls.NewFunction("has_group",
		decls.NewOverload("has_group_string",
			[]*exprpb.Type{decls.String},
			decls.Bool))
	fList, dList := &functions.Overload{
		Operator: "has_group_string_list",
		Unary: func(values ref.Val) ref.Val {
			list := values.Value().([]ref.Val)
			for _, group := range list {
				g := group.Value().(string)
				if hasGroup(g, host.GencfgGroups) {
					return types.Bool(true)
				}
			}
			return types.Bool(false)
		},
	}, decls.NewFunction("has_group",
		decls.NewOverload("has_group_string_list",
			[]*exprpb.Type{decls.NewListType(decls.String)},
			decls.Bool))
	return append(funcs, f, fList), append(declarations, d, dList)
}

func GeoFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	geo := func(loc string, location string) bool {
		return loc == location
	}
	f, d := &functions.Overload{
		Operator: "geo_string",
		Unary: func(lhs ref.Val) ref.Val {
			val := lhs.Value().(string)
			return types.Bool(geo(val, host.Location))
		},
	}, decls.NewFunction("geo",
		decls.NewOverload("geo_string",
			[]*exprpb.Type{decls.String},
			decls.Bool))
	fList, dList := &functions.Overload{
		Operator: "geo_string_list",
		Unary: func(values ref.Val) ref.Val {
			list := values.Value().([]ref.Val)
			for _, proj := range list {
				p := proj.Value().(string)
				if geo(p, host.Location) {
					return types.Bool(true)
				}
			}
			return types.Bool(false)
		},
	}, decls.NewFunction("geo",
		decls.NewOverload("geo_string_list",
			[]*exprpb.Type{decls.NewListType(decls.String)},
			decls.Bool))
	return append(funcs, f, fList), append(declarations, d, dList)
}

func PrestableFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	prestable := func(walleTags []string) bool {
		return slices.ContainsString(walleTags, "rtc.stage-prestable")
	}
	f, d := &functions.Overload{
		Operator: "prestable",
		Function: func(values ...ref.Val) ref.Val {
			return types.Bool(prestable(host.WalleTags))
		},
	}, decls.NewFunction("prestable",
		decls.NewOverload("prestable",
			[]*exprpb.Type{},
			decls.Bool))
	return append(funcs, f), append(declarations, d)
}

func ProductionFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	production := func(walleTags []string) bool {
		return slices.ContainsString(walleTags, "rtc.stage-production")
	}
	f, d := &functions.Overload{
		Operator: "production",
		Function: func(values ...ref.Val) ref.Val {
			return types.Bool(production(host.WalleTags))
		},
	}, decls.NewFunction("production",
		decls.NewOverload("production",
			[]*exprpb.Type{},
			decls.Bool))
	return append(funcs, f), append(declarations, d)
}

func ProjectFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	project := func(p, walleProject string) bool {
		return p == walleProject
	}
	f, d := &functions.Overload{
		Operator: "prj_string",
		Unary: func(lhs ref.Val) ref.Val {
			val := lhs.Value().(string)
			return types.Bool(project(val, host.WalleProject))
		},
	}, decls.NewFunction("prj",
		decls.NewOverload("prj_string",
			[]*exprpb.Type{decls.String},
			decls.Bool))
	fList, dList := &functions.Overload{
		Operator: "prj_string_list",
		Unary: func(values ref.Val) ref.Val {
			list := values.Value().([]ref.Val)
			for _, proj := range list {
				p := proj.Value().(string)
				if project(p, host.WalleProject) {
					return types.Bool(true)
				}
			}
			return types.Bool(false)
		},
	}, decls.NewFunction("prj",
		decls.NewOverload("prj_string_list",
			[]*exprpb.Type{decls.NewListType(decls.String)},
			decls.Bool))
	return append(funcs, f, fList), append(declarations, d, dList)
}

func DCQueueFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	dcQueue := func(q, dcQueue string) bool {
		return q == dcQueue
	}
	f, d := &functions.Overload{
		Operator: "dc_queue_string",
		Unary: func(lhs ref.Val) ref.Val {
			val := lhs.Value().(string)
			return types.Bool(dcQueue(val, host.DcQueue))
		},
	}, decls.NewFunction("dc_queue",
		decls.NewOverload("dc_queue_string",
			[]*exprpb.Type{decls.String},
			decls.Bool))
	fList, dList := &functions.Overload{
		Operator: "dc_queue_string_list",
		Unary: func(values ref.Val) ref.Val {
			list := values.Value().([]ref.Val)
			for _, proj := range list {
				p := proj.Value().(string)
				if dcQueue(p, host.DcQueue) {
					return types.Bool(true)
				}
			}
			return types.Bool(false)
		},
	}, decls.NewFunction("dc_queue",
		decls.NewOverload("dc_queue_string_list",
			[]*exprpb.Type{decls.NewListType(decls.String)},
			decls.Bool))
	return append(funcs, f, fList), append(declarations, d, dList)
}

func PercFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	perc := func(v, num int64) bool {
		// Percents counts as 1..100, but server_info/genServerNum() produce 0..99.
		// Strict less is used to align the difference.
		return num < v
	}
	f, d := &functions.Overload{
		Operator: "perc_int",
		Unary: func(lhs ref.Val) ref.Val {
			val := lhs.Value().(int64)
			return types.Bool(perc(val, int64(host.Num)))
		},
	}, decls.NewFunction("perc",
		decls.NewOverload("perc_int",
			[]*exprpb.Type{decls.Int},
			decls.Bool))
	return append(funcs, f), append(declarations, d)
}

func HostFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	hostF := func(host, hostname string) bool {
		if !strings.Contains(host, ".") {
			host += ".search.yandex.net"
		}
		return hostname == host
	}
	f, d := &functions.Overload{
		Operator: "h_string",
		Unary: func(lhs ref.Val) ref.Val {
			val := lhs.Value().(string)
			return types.Bool(hostF(val, host.Hostname))
		},
	}, decls.NewFunction("h",
		decls.NewOverload("h_string",
			[]*exprpb.Type{decls.String},
			decls.Bool))
	fList, dList := &functions.Overload{
		Operator: "h_string_list",
		Unary: func(values ref.Val) ref.Val {
			list := values.Value().([]ref.Val)
			for _, hostname := range list {
				h := hostname.Value().(string)
				if hostF(h, host.Hostname) {
					return types.Bool(true)
				}
			}
			return types.Bool(false)
		},
	}, decls.NewFunction("h",
		decls.NewOverload("h_string_list",
			[]*exprpb.Type{decls.NewListType(decls.String)},
			decls.Bool))
	return append(funcs, f, fList), append(declarations, d, dList)
}

func SwitchFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	switchF := func(v, netSwitch string) bool {
		return netSwitch == v
	}
	f, d := &functions.Overload{
		Operator: "switch_string",
		Unary: func(lhs ref.Val) ref.Val {
			val := lhs.Value().(string)
			return types.Bool(switchF(val, host.NetSwitch))
		},
	}, decls.NewFunction("switch",
		decls.NewOverload("switch_string",
			[]*exprpb.Type{decls.String},
			decls.Bool))
	fList, dList := &functions.Overload{
		Operator: "switch_string_list",
		Unary: func(values ref.Val) ref.Val {
			list := values.Value().([]ref.Val)
			for _, sw := range list {
				s := sw.Value().(string)
				if switchF(s, host.NetSwitch) {
					return types.Bool(true)
				}
			}
			return types.Bool(false)
		},
	}, decls.NewFunction("switch",
		decls.NewOverload("switch_string_list",
			[]*exprpb.Type{decls.NewListType(decls.String)},
			decls.Bool))
	return append(funcs, f, fList), append(declarations, d, dList)
}

func DefaultFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	f, d := &functions.Overload{
		Operator: "default",
		Function: func(values ...ref.Val) ref.Val {
			return types.Bool(true)
		},
	}, decls.NewFunction("default",
		decls.NewOverload("default",
			[]*exprpb.Type{},
			decls.Bool))
	return append(funcs, f), append(declarations, d)
}

func FocalFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	focal := func(osCodename string) bool {
		return osCodename == "focal"
	}
	f, d := &functions.Overload{
		Operator: "focal",
		Function: func(values ...ref.Val) ref.Val {
			return types.Bool(focal(host.OsCodename))
		},
	}, decls.NewFunction("focal",
		decls.NewOverload("focal",
			[]*exprpb.Type{},
			decls.Bool))
	return append(funcs, f), append(declarations, d)
}

func XenialFunc(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	xenial := func(osCodename string) bool {
		return osCodename == "xenial"
	}
	f, d := &functions.Overload{
		Operator: "xenial",
		Function: func(values ...ref.Val) ref.Val {
			return types.Bool(xenial(host.OsCodename))
		},
	}, decls.NewFunction("xenial",
		decls.NewOverload("xenial",
			[]*exprpb.Type{},
			decls.Bool))
	return append(funcs, f), append(declarations, d)
}

func AMD64Func(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	amd64 := func(osArch string) bool {
		return osArch == "amd64"
	}
	f, d := &functions.Overload{
		Operator: "amd64",
		Function: func(values ...ref.Val) ref.Val {
			return types.Bool(amd64(host.OsArch))
		},
	}, decls.NewFunction("amd64",
		decls.NewOverload("amd64",
			[]*exprpb.Type{},
			decls.Bool))
	return append(funcs, f), append(declarations, d)
}

func ARM64Func(host *pb.HostInfo, funcs []*functions.Overload, declarations []*exprpb.Decl) ([]*functions.Overload, []*exprpb.Decl) {
	arm64 := func(osArch string) bool {
		return osArch == "arm64"
	}
	f, d := &functions.Overload{
		Operator: "arm64",
		Function: func(values ...ref.Val) ref.Val {
			return types.Bool(arm64(host.OsArch))
		},
	}, decls.NewFunction("arm64",
		decls.NewOverload("arm64",
			[]*exprpb.Type{},
			decls.Bool))
	return append(funcs, f), append(declarations, d)
}
