package environment

import (
	"io/fs"
	"os"
	"path"
)

const (
	runHostman    = "run/hostman" // Runtime userOverrides
	etcHostman    = "etc/hostman" // Persistent unit userOverrides
	unitsD        = "units.d"
	portoDaemonsD = "porto-daemons.d"

	repoCurrent = "var/lib/ya-salt/repo/current"
)

var hostmanDirs = []string{etcHostman, runHostman}

// Path hostctl paths. Returns only existing directories.
// Returns in order from least priority to most priority
//
// e.g
// * /var/lib/ya-salt/repo/current/*/units.d
// * /var/lib/ya-salt/repo/current/*/porto-daemons.d
// * /etc/hostman/units.d
// * /etc/hostman/porto-daemons.d
// * /run/hostman/units.d
// * /run/hostman/porto-daemons.d
// * /<repo-override>/*/units.d
// * /<repo-override>/*/porto-daemons.d
// * cmdline userOverrides
func Path(fsys fs.FS, overrides []string, repoOverride string) []string {
	if !path.IsAbs(repoOverride) {
		// make abs path
		cwd, err := os.Getwd()
		if err != nil {
			panic(err)
		}
		repoOverride = path.Join(cwd, repoOverride)
	}
	// remove first symbol: always '/'
	repoOverride = repoOverride[1:]
	// remove last '/' if exist
	if len(repoOverride) > 0 && repoOverride[len(repoOverride)-1] == '/' {
		repoOverride = repoOverride[:len(repoOverride)-1]
	}
	return discover(fsys, overrides, repoOverride)
}

func productionOrigins(rootFS fs.FS, repos []string) []string {
	dirs := make([]string, 0)
	for _, repo := range repos {
		dirs = append(dirs, repoDirs(rootFS, repo)...)
	}
	dirs = append(dirs, hostmanDirs...)
	origins := make([]string, 0)
	for _, name := range dirs {
		p := path.Join(name, unitsD)
		if isDir(rootFS, p) {
			origins = append(origins, p)
		}
		p = path.Join(name, portoDaemonsD)
		if isDir(rootFS, p) {
			origins = append(origins, p)
		}
	}
	return origins
}

func repoDirs(rootFS fs.FS, repoOverride string) []string {
	// add all dirs from '/var/lib/ya-salt/repo/current' to names. env.g. core,sysdev
	files, err := fs.ReadDir(rootFS, repoOverride)
	if err != nil {
		return nil
	}
	names := make([]string, len(files))
	for i := 0; i < len(files); i++ {
		names[i] = path.Join(repoOverride, files[i].Name())
	}
	return names
}

func isDir(rootFS fs.FS, name string) bool {
	stat, err := fs.Stat(rootFS, name)
	return err == nil && stat.IsDir()
}

func discover(rootFS fs.FS, overrides []string, repoOverride string) []string {
	// Fallback current to repo dirs.
	// We can call ya-salt run --local with only one repo,
	// and we'll miss a lot units
	repos := []string{repoCurrent}
	if repoOverride != repoCurrent {
		repos = append(repos, repoOverride)
	}
	origins := productionOrigins(rootFS, repos)
	for i := 0; i < len(origins); i++ {
		origins[i] = setRoot(origins[i])
	}
	// do not modify root for cmdline dirs userOverrides
	origins = append(origins, overrides...)
	return origins
}

// Append '/' in the path beginning
func setRoot(p string) string {
	return path.Join("/", p)
}
