package environment

import (
	"github.com/stretchr/testify/assert"
	"io/fs"
	"path"
	"testing"
	"testing/fstest"
)

func Test_discoverPaths(t *testing.T) {
	const repoCurrent = "var/lib/ya-salt/repo/current"
	tests := []struct {
		name      string
		makeFs    func(fstest.MapFS)
		overrides []string
		want      []string
		repo      string
	}{{
		name: "/etc/hostman",
		makeFs: func(f fstest.MapFS) {
			f[path.Join(etcHostman, unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join(etcHostman, portoDaemonsD)] = &fstest.MapFile{Mode: fs.ModeDir}
		},
		repo:      repoCurrent,
		overrides: make([]string, 0),
		want: []string{
			setRoot(path.Join(etcHostman, unitsD)),
			setRoot(path.Join(etcHostman, portoDaemonsD)),
		},
	}, {
		name: "/run/hostman",
		makeFs: func(f fstest.MapFS) {
			f[path.Join(runHostman, unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join(runHostman, portoDaemonsD)] = &fstest.MapFile{Mode: fs.ModeDir}
		},
		repo:      repoCurrent,
		overrides: make([]string, 0),
		want: []string{
			setRoot(path.Join(runHostman, unitsD)),
			setRoot(path.Join(runHostman, portoDaemonsD)),
		},
	}, {
		name: "/run/hostman /etc/hostman",
		makeFs: func(f fstest.MapFS) {
			f[path.Join(runHostman, unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join(runHostman, portoDaemonsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join(etcHostman, unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join(etcHostman, portoDaemonsD)] = &fstest.MapFile{Mode: fs.ModeDir}
		},
		repo:      repoCurrent,
		overrides: make([]string, 0),
		want: []string{
			setRoot(path.Join(etcHostman, unitsD)),
			setRoot(path.Join(etcHostman, portoDaemonsD)),
			setRoot(path.Join(runHostman, unitsD)),
			setRoot(path.Join(runHostman, portoDaemonsD)),
		},
	}, {
		name: "userOverrides /run/hostman",
		makeFs: func(f fstest.MapFS) {
			f[path.Join(runHostman, unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join(runHostman, portoDaemonsD)] = &fstest.MapFile{Mode: fs.ModeDir}
		},
		repo:      repoCurrent,
		overrides: []string{unitsD},
		want: []string{
			setRoot(path.Join(runHostman, unitsD)),
			setRoot(path.Join(runHostman, portoDaemonsD)),
			path.Join(unitsD),
		},
	}, {
		name: "/etc/hostman core repo",
		makeFs: func(f fstest.MapFS) {
			f[path.Join(etcHostman, unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join(etcHostman, portoDaemonsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join(repoCurrent, "core", unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
		},
		repo:      repoCurrent,
		overrides: make([]string, 0),
		want: []string{
			setRoot(path.Join(repoCurrent, "core", unitsD)),
			setRoot(path.Join(etcHostman, unitsD)),
			setRoot(path.Join(etcHostman, portoDaemonsD)),
		},
	}, {
		name: "repo override",
		makeFs: func(f fstest.MapFS) {
			f[path.Join(repoCurrent, "core", unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
			f[path.Join("repo", "core", unitsD)] = &fstest.MapFile{Mode: fs.ModeDir}
		},
		repo:      "repo",
		overrides: make([]string, 0),
		want: []string{
			setRoot(path.Join(repoCurrent, "core", unitsD)),
			setRoot(path.Join("repo", "core", unitsD)),
		},
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			f := fstest.MapFS{}
			tt.makeFs(f)
			for s := range f {
				if ok := fs.ValidPath(s); !ok {
					t.Errorf("invalid path '%s'", s)
				}
			}
			got := discover(f, tt.overrides, tt.repo)
			assert.Equal(t, tt.want, got)
		})
	}
}
