package environment

import (
	"a.yandex-team.ru/infra/hostctl/internal/engine/definitions"
	"a.yandex-team.ru/library/go/core/log"
	"bytes"
	"errors"
	"fmt"
	"io"
	"io/fs"
	"os"
	"path"
	"path/filepath"
	"strings"
)

var ErrNotYAMLFile = errors.New("file is not yaml")

/*
Read units definitions from origins
  * If names was defined reads only needed files for this units.
    Skips files not matching pattern {name}.yaml or {name}.yml
  * If names was nil reads definitions from all files with .yaml or .yml ext
*/
func Read(l log.Logger, fsys fs.FS, origins []string, names []string) map[string][]definitions.NamedReader {
	discovered := make(map[string][]definitions.NamedReader)
	for _, origin := range origins {
		o, err := makeAbs(origin)
		if err != nil {
			l.Errorf("Failed to get abs path for %s: %s", origin, err)
		}
		entries, err := fs.ReadDir(fsys, o)
		if err != nil {
			l.Warnf("Failed to read %s origin: %s", origin, err)
		}
		for _, entry := range entries {
			n := entry.Name()
			if names == nil {
				if path.Ext(n) != ".yaml" && path.Ext(n) != ".yml" {
					l.Debugf("Skipping %s definition from %s", n, origin)
					continue
				}
			} else {
				ok := false
				for _, name := range names {
					if n == name+".yaml" || n == name+".yml" {
						ok = true
					}
				}
				if !ok {
					l.Debugf("Skipping %s definition from %s", n, origin)
					continue
				}
			}
			name := entry.Name()
			p := path.Join(origin, name)
			l.Infof("Reading %s definition from %s", name, origin)
			buf, err := ReadFromPath(fsys, p)
			if err == nil {
				unitName := strings.TrimSuffix(name, filepath.Ext(name))
				def := definitions.New(buf, p)
				if _, ok := discovered[unitName]; !ok {
					discovered[unitName] = make([]definitions.NamedReader, 0, 1)
				}
				discovered[unitName] = append(discovered[unitName], def)
			} else if !errors.Is(err, ErrNotYAMLFile) {
				l.Warnf("Failed to read %s: %s", p, err)
			}
		}
	}
	return discovered
}

func makeAbs(p string) (string, error) {
	if !path.IsAbs(p) {
		wd, err := os.Getwd()
		if err != nil {
			return "", err
		}
		p = path.Join(wd, p)
	}
	return p[1:], nil
}

func ReadFromPath(rootFS fs.FS, p string) (io.Reader, error) {
	if path.Ext(p) != ".yaml" && path.Ext(p) != ".yml" {
		return nil, fmt.Errorf("%s: %w", p, ErrNotYAMLFile)
	}
	f, err := rootFS.Open(rootFSPath(p))
	if err != nil {
		return nil, err
	}
	defer func() { _ = f.Close() }()
	buf := &bytes.Buffer{}
	if _, err := io.Copy(buf, f); err != nil {
		return nil, err
	}
	return buf, nil
}

func rootFSPath(p string) string {
	if !path.IsAbs(p) {
		cwd, err := os.Getwd()
		if err != nil {
			panic(fmt.Errorf("failed to get cwd: %w", err))
		}
		p = path.Join(cwd, p)
	}
	return p[1:]
}
