package environment

import (
	"a.yandex-team.ru/infra/hostctl/internal/engine/definitions"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"github.com/stretchr/testify/assert"
	"io"
	"io/fs"
	"path"
	"testing"
	"testing/fstest"
)

func TestUnit(t *testing.T) {
	type testResult struct {
		origin  string
		content string
	}
	makeTestResult := func(d definitions.NamedReader) testResult {
		c, _ := io.ReadAll(d)
		return testResult{
			origin:  d.Name(),
			content: string(c),
		}
	}
	const repoCurrent = "var/lib/ya-salt/repo/current"
	coreOrigin := path.Join(repoCurrent, "core", "units.d")
	sysdevOrigin := path.Join(repoCurrent, "sysdev", "units.d")
	etcOrigin := path.Join("etc", "hostman", "units.d")
	origins := []string{"/" + coreOrigin, "/" + sysdevOrigin, "/" + etcOrigin}
	tests := []struct {
		name   string
		makeFS func(fstest.MapFS)
		unit   string
		want   []testResult
	}{{
		name: "single unit",
		makeFS: func(f fstest.MapFS) {
			f[path.Join(coreOrigin, "apt.yaml")] = &fstest.MapFile{Mode: fs.FileMode(0666), Data: []byte("apt from core")}
		},
		unit: "apt",
		want: []testResult{{
			origin:  "/" + coreOrigin + "/apt.yaml",
			content: "apt from core",
		}},
	}, {
		name: "overridden unit",
		makeFS: func(f fstest.MapFS) {
			f[path.Join(coreOrigin, "apt.yaml")] = &fstest.MapFile{Mode: fs.FileMode(0666), Data: []byte("apt from core")}
			f[path.Join(etcOrigin, "apt.yaml")] = &fstest.MapFile{Mode: fs.FileMode(0666), Data: []byte("apt from etc")}
		},
		unit: "apt",
		want: []testResult{{
			origin:  "/" + coreOrigin + "/apt.yaml",
			content: "apt from core",
		}, {
			origin:  "/" + etcOrigin + "/apt.yaml",
			content: "apt from etc",
		}},
	}, {
		name: "invalid name",
		makeFS: func(f fstest.MapFS) {
			f[path.Join(coreOrigin, "definitely-not-apt.yaml")] = &fstest.MapFile{Mode: fs.FileMode(0666), Data: []byte("definitely not apt from core")}
		},
		unit: "apt",
		want: make([]testResult, 0),
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			cfg := zap.CLIConfig(log.DebugLevel)
			cfg.Development = true
			l, _ := zap.New(cfg)
			f := fstest.MapFS{}
			tt.makeFS(f)
			got := Read(l, f, origins, []string{tt.unit})[tt.unit]
			result := make([]testResult, len(got))
			for i, d := range got {
				result[i] = makeTestResult(d)
			}
			assert.Equal(t, tt.want, result)
		})
	}
}
