package render

import (
	"a.yandex-team.ru/infra/hostctl/internal/engine/hostctx"
	"a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	"a.yandex-team.ru/infra/hostctl/pkg/pbutil"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"fmt"
	"google.golang.org/protobuf/proto"
)

type CompiledTemplate struct {
	// TODO: here will results of cel.env compilation
	meta *pb.ObjectMeta
	spec proto.Message
	kind kind.Kind
}

type evaluatedTemplate struct {
	compiled *CompiledTemplate
	ctxMap   map[string]string
}

func CompileTemplate(meta *pb.ObjectMeta, spec proto.Message, kind kind.Kind) (*CompiledTemplate, error) {
	return &CompiledTemplate{meta, spec, kind}, nil
}

func (t *CompiledTemplate) EvalCtx(hi *pb.HostInfo) (*evaluatedTemplate, error) {
	// Eval context deriving substitutions
	ctxMap, err := hostctx.Eval(hi, t.meta.Ctx)
	if err != nil {
		return nil, fmt.Errorf("failed to eval ctx on '%s': %w", t.meta.Name, err)
	}
	return &evaluatedTemplate{
		compiled: t,
		ctxMap:   ctxMap,
	}, nil
}

func (e *evaluatedTemplate) ApplyCtx() (*unit.Unit, error) {
	if err := hostctx.ApplyCtx(e.compiled.meta, e.compiled.spec, e.ctxMap); err != nil {
		return nil, fmt.Errorf("failed to apply ctx on '%s': %w", e.compiled.meta.Name, err)
	}
	id, err := pbutil.PbDigest(e.compiled.spec)
	if err != nil {
		return nil, fmt.Errorf("failed to create id for '%s': %w", e.compiled.meta.Name, err)
	}
	s := &pb.SkipRemovePhase{}
	if forget := e.compiled.meta.Annotations["skip-remove-phase"]; len(forget) != 0 {
		if err := yamlutil.UnmarshalStrict([]byte(forget), s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(meta.annotation.forget): %w", err)
		}
	}
	return unit.RenderedUnit(
		e.compiled.spec,
		id, e.compiled.meta.Name, e.compiled.kind,
		&pb.SlotMeta{
			Labels:          e.compiled.meta.Labels,
			Annotations:     e.compiled.meta.Annotations,
			XKind:           e.compiled.meta.Kind,
			SkipRemovePhase: s,
		}, &pb.RevisionMeta{
			Kind:    e.compiled.meta.Kind,
			Version: e.compiled.meta.Version,
		}, absent(e.compiled.meta)), nil
}

func absent(m *pb.ObjectMeta) bool {
	return m.DeleteRequested || m.Annotations["stage"] == "absent"
}
