package template

import (
	"errors"
	"fmt"
	"io"

	"a.yandex-team.ru/infra/hostctl/internal/engine/render"
	"a.yandex-team.ru/infra/hostctl/internal/unit"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/hostctl/internal/engine/definitions"
	"a.yandex-team.ru/infra/hostctl/internal/engine/expire"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"google.golang.org/protobuf/encoding/protojson"
)

func Removed(name string) *Template {
	return &Template{
		meta: &pb.ObjectMeta{
			Kind:        "PackageSet", // do not care
			Name:        name,
			Ctx:         &pb.Context{},
			Annotations: map[string]string{"stage": "absent"},
		},
		// do not care
		spec: &pb.PackageSetSpec{},
	}
}

// Template is opaque object which can be rendered into a unit. It has no ID, no public meta field.
type Template struct {
	meta *pb.ObjectMeta
	spec proto.Message
}

/*
Render applies provided host info and returns Unit and Meta describing actions to be applied to slot.
Can be called multiple times with different host info attributes.
*/
func (t *Template) Render(hi *pb.HostInfo) (*unit.Unit, error) {
	compiled, err := t.Compile()
	if err != nil {
		return nil, err
	}
	evalResult, err := compiled.EvalCtx(hi)
	if err != nil {
		return nil, err
	}
	return evalResult.ApplyCtx()
}

// Special Unmarshaler implementations to decode protobuf types in anonymous structs used in fromYAMLs.
type metaValue pb.ObjectMeta

func (m *metaValue) UnmarshalJSON(b []byte) error {
	return protojson.Unmarshal(b, (*pb.ObjectMeta)(m))
}

type specPackageSetValue pb.PackageSetSpec

func (m *specPackageSetValue) UnmarshalJSON(b []byte) error {
	return protojson.Unmarshal(b, (*pb.PackageSetSpec)(m))
}

type specSystemServiceValue pb.SystemServiceSpec

func (m *specSystemServiceValue) UnmarshalJSON(b []byte) error {
	return protojson.Unmarshal(b, (*pb.SystemServiceSpec)(m))
}

type specTimerJobValue pb.TimerJobSpec

func (m *specTimerJobValue) UnmarshalJSON(b []byte) error {
	return protojson.Unmarshal(b, (*pb.TimerJobSpec)(m))
}

type specPortoDaemonValue pb.PortoDaemon

func (m *specPortoDaemonValue) UnmarshalJSON(b []byte) error {
	return protojson.Unmarshal(b, (*pb.PortoDaemon)(m))
}

type specPodValue pb.HostPodSpec

func (m *specPodValue) UnmarshalJSON(b []byte) error {
	return protojson.Unmarshal(b, (*pb.HostPodSpec)(m))
}

func Parse(def definitions.NamedReader) (*Template, error) {
	ctxBytes, objBytes, err := splitYAML(def)
	if err != nil {
		return nil, fmt.Errorf("failed to read defenition: %w", err)
	}
	return fromYAMLs(ctxBytes, objBytes, def.Name())
}

// splitYAML scans provided reader and extracts at most two YAML documents:
// Thus file can look like:
// vars:
//     ...
// --- <- new object in YAML syntax
// meta:
//     ...
// spec:
//     ...
func splitYAML(f io.Reader) (meta []byte, spec []byte, err error) {
	s := yamlutil.NewDocumentDecoder(f).Scanner()
	documents := make([][]byte, 0)
	for s.Scan() {
		// Scanner overwrites underlying buffer, so we need a copy instead of slice
		doc := make([]byte, len(s.Bytes()))
		copy(doc, s.Bytes())
		documents = append(documents, doc)
	}
	if len(documents) <= 0 {
		return nil, nil, errors.New("no YAML documents")
	}
	if len(documents) > 2 {
		return nil, nil, fmt.Errorf("too many YAML documents: %d, "+
			"need 1 (obj) or 2 (ctx + obj)", len(documents))
	}
	if len(documents) == 2 {
		return documents[0], documents[1], nil
	}
	return nil, documents[0], nil
}

// fromYAMLs decodes ctxBytes and objBytes documents to pb.ObjMeta (which includes pb.Context from cxtBytes)
// and one of available specs.
// pb.ObjMeta and spec objects are decoded from the same objBytes document using anonymous structs to extract
// `spec` and `meta` fields from yaml document.
func fromYAMLs(ctxBytes, objBytes []byte, path string) (*Template, error) {
	meta := &pb.ObjectMeta{}
	err := unmarshalMeta(objBytes, meta)
	if err != nil {
		return nil, err
	}
	ctx := &pb.Context{
		Vars: make([]*pb.Context_Var, 0),
	}
	if ctxBytes != nil {
		err = yamlutil.UnmarshalStrict(ctxBytes, ctx)
		if err != nil {
			return nil, err
		}
	}
	meta.Ctx = ctx
	if meta.Annotations == nil {
		meta.Annotations = make(map[string]string)
	}
	meta.Annotations["filename"] = path
	if len(meta.Annotations["stage"]) == 0 {
		// Fill in stage manually, was not required at some point.
		// For backward compatibility.
		meta.Annotations["stage"] = "<default>"
	}
	// Just validation. Fall on unsuccessful result
	if forget := meta.Annotations["skip-remove-phase"]; len(forget) != 0 {
		f := &pb.SkipRemovePhase{}
		err := yamlutil.UnmarshalStrict([]byte(forget), f)
		if err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(meta.annotation.forget): %w", err)
		}
	}
	switch kind.Kind(meta.Kind) {
	case kind.PackageSet:
		spec := &pb.PackageSetSpec{}
		s := &struct {
			Spec *specPackageSetValue `yaml:"spec"`
		}{Spec: (*specPackageSetValue)(spec)}
		if err := yamlutil.UnmarshalStrict(objBytes, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return &Template{meta, spec}, nil
	case kind.PortoDaemon:
		spec := &pb.PortoDaemon{}
		s := &struct {
			Spec *specPortoDaemonValue `yaml:"spec"`
		}{Spec: (*specPortoDaemonValue)(spec)}
		if err := yamlutil.UnmarshalStrict(objBytes, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return &Template{meta, spec}, nil
	case kind.SystemService:
		spec := &pb.SystemServiceSpec{}
		s := &struct {
			Spec *specSystemServiceValue `yaml:"spec"`
		}{Spec: (*specSystemServiceValue)(spec)}
		if err := yamlutil.UnmarshalStrict(objBytes, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return &Template{meta, spec}, nil
	case kind.TimerJob:
		spec := &pb.TimerJobSpec{}
		s := &struct {
			Spec *specTimerJobValue `yaml:"spec"`
		}{Spec: (*specTimerJobValue)(spec)}
		if err := yamlutil.UnmarshalStrict(objBytes, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return &Template{meta, spec}, nil
	case kind.HostPod:
		spec := &pb.HostPodSpec{}
		s := &struct {
			Spec *specPodValue `yaml:"spec"`
		}{Spec: (*specPodValue)(spec)}
		if err := yamlutil.UnmarshalStrict(objBytes, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return &Template{meta, spec}, nil
	}
	return nil, fmt.Errorf("invalid unit kind: %s, possible kinds: [%s]", meta.Kind, kind.Kinds)
}

// NewTestTemplate creates new template without validation, designed to be used in tests only.
func NewTestTemplate(meta *pb.ObjectMeta, spec proto.Message) *Template {
	return &Template{meta, spec}
}

func (t *Template) Kind() kind.Kind {
	return kind.Kind(t.meta.Kind)
}

func (t *Template) Name() string {
	return t.meta.Name
}

func (t *Template) Path() string {
	return t.meta.Annotations["filename"]
}

func (t *Template) Copy() (*pb.ObjectMeta, proto.Message) {
	m := &pb.ObjectMeta{}
	proto.Merge(m, t.meta)
	var spec proto.Message
	switch t.Kind() {
	case kind.TimerJob:
		spec = &pb.TimerJobSpec{}
	case kind.PackageSet:
		spec = &pb.PackageSetSpec{}
	case kind.PortoDaemon:
		spec = &pb.PortoDaemon{}
	case kind.SystemService:
		spec = &pb.SystemServiceSpec{}
	case kind.HostPod:
		spec = &pb.HostPodSpec{}
	default:
		panic("unknown template kind")
	}
	proto.Merge(spec, t.spec)
	return m, spec
}

func (t *Template) Expired() (bool, error) {
	return expire.Expired(t.meta)
}

func (t *Template) Compile() (*render.CompiledTemplate, error) {
	return render.CompileTemplate(t.meta, t.spec, t.Kind())
}

func unmarshalMeta(buf []byte, meta *pb.ObjectMeta) error {
	m := &struct {
		Meta *metaValue `yaml:"meta"`
	}{
		Meta: (*metaValue)(meta),
	}
	if err := yamlutil.UnmarshalStrict(buf, m); err != nil {
		return fmt.Errorf("can not unmarshall YAML(meta): %w", err)
	}
	return nil
}
