package template

import (
	"bytes"
	"fmt"
	"strings"
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/engine/definitions"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/test/assertpb"
)

func Test_absent(t *testing.T) {
	tests := []struct {
		name string
		init func(m *pb.ObjectMeta)
		want bool
	}{{
		name: "delete_requested",
		init: func(m *pb.ObjectMeta) {
			m.DeleteRequested = true
			m.Annotations["stage"] = "stable"
		},
		want: true,
	}, {
		name: "stage absent",
		init: func(m *pb.ObjectMeta) {
			m.DeleteRequested = false
			m.Annotations["stage"] = "absent"
		},
		want: true,
	}, {
		name: "stage absent && delete_requested",
		init: func(m *pb.ObjectMeta) {
			m.DeleteRequested = true
			m.Annotations["stage"] = "absent"
		},
		want: true,
	}, {
		name: "stage stable",
		init: func(m *pb.ObjectMeta) {
			m.DeleteRequested = false
			m.Annotations["stage"] = "stable"
		},
		want: false,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			m := &pb.ObjectMeta{
				Kind:            "PackageSet",
				Name:            "test",
				DeleteRequested: false,
				Annotations:     make(map[string]string),
			}
			tt.init(m)
			if got := absent(m); got != tt.want {
				t.Errorf("absent() = %v, want %v", got, tt.want)
			}
		})
	}
}

const (
	ctxYAML = `include:
vars:
 - name: ver
   match:
    - exp: geo('sas')
      val: 1.2.32
    - exp: prestable()
      val: 1`
	packageSetYAML = `
meta:
  kind: PackageSet
  name: unit
  version: '2'
  delete_requested: False
  labels:
    test: 'test'
  annotations:
    env.noop: 'True'
spec:
  packages:
    - name: apt
      version: '{ver}'
  files:
    - path: /tmp/file1
      content: test content
      user: root
      group: root
      mode: "0644"`
	packageSetWithSkipRemoveYAML = `
meta:
  kind: PackageSet
  name: unit
  version: '2'
  delete_requested: False
  labels:
    test: 'test'
  annotations:
    env.noop: 'True'
    skip-remove-phase: |+
      packages:
      - apt
spec:
  packages:
    - name: apt
      version: '{ver}'
  files:
    - path: /tmp/file1
      content: test content
      user: root
      group: root
      mode: "0644"`
	portoDaemonYAML = `
meta:
  kind: "PortoDaemon"
  version: "1"
  name: "example-http-serv"
  delete_requested: false
spec:
  properties:
    virt_mode: "host"
    cmd:
      - "python -m SimpleHTTPServer 8000"
    isolate: "false"
    user: "root"
    group: "root"
    controllers:
      devices: "false"
    memory_guarantee: "600Mb"
    memory_limit: "2Gb"
    cpu_guarantee: "0.5c"
    cpu_limit: "3c"`
	portoDaemonWithInvalidKindYAML = `
meta:
  kind: "InvalidKind"
  version: "1"
  name: "example-http-serv"
  delete_requested: false
spec:
  properties:
    virt_mode: "host"
    cmd:
      - "python -m SimpleHTTPServer 8000"
    isolate: "false"
    user: "root"
    group: "root"
    controllers:
      devices: "false"
    memory_guarantee: "600Mb"
    memory_limit: "2Gb"
    cpu_guarantee: "0.5c"
    cpu_limit: "3c"`
	portoDaemonYAMLWithMisspelledFields = `
meta:
  kind: "PortoDaemon"
  version: "1"
  name: "example-http-serv"
  delete_requested: false
spec:
  properties:
    virt_mode: "host"
    cmd:
      - "python -m SimpleHTTPServer 8000"
    isolate: "false"
    user: "root"
    group: "root"
    controllers:
      devices: "false"
    memory_guaranty: "600Mb"
    memory_limit: "2Gb"
    cpu_guaranty: "0.5c"
    cpu_limit: "3c"`
)

var (
	ctxPb = &pb.Context{
		Vars: []*pb.Context_Var{{
			Name: "ver",
			Match: []*pb.Context_Match{{
				Exp: "geo('sas')",
				Val: "1.2.32",
			}, {
				Exp: "prestable()",
				Val: "1",
			}},
		}},
	}
	packageSetMeta = &pb.ObjectMeta{
		Kind:            "PackageSet",
		Name:            "unit",
		Version:         "2",
		DeleteRequested: false,
		Labels:          map[string]string{"test": "test"},
		Annotations:     map[string]string{"env.noop": "True", "filename": "/specs/PackageSet.yaml", "stage": "<default>"},
	}
	packageSetMetaWithSkipRemove = &pb.ObjectMeta{
		Kind:            "PackageSet",
		Name:            "unit",
		Version:         "2",
		DeleteRequested: false,
		Labels:          map[string]string{"test": "test"},
		Annotations: map[string]string{
			"env.noop": "True",
			"filename": "/specs/PackageSet.yaml",
			"stage":    "<default>", "skip-remove-phase": "packages:\n- apt\n"},
	}
	packageSetSpec = &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{{
			Name:    "apt",
			Version: "{ver}",
		}},
		Files: []*pb.ManagedFile{{
			Path:    "/tmp/file1",
			Content: "test content",
			User:    "root",
			Group:   "root",
			Mode:    "0644",
		}},
	}
	portoDaemonMeta = &pb.ObjectMeta{
		Kind:            "PortoDaemon",
		Name:            "example-http-serv",
		Version:         "1",
		DeleteRequested: false,
		Annotations:     map[string]string{"filename": "/specs/PortoDaemon.yaml", "stage": "<default>"},
	}
	portoDaemonSpec = &pb.PortoDaemon{
		Properties: &pb.PortoProperties{
			Cmd:             []string{"python -m SimpleHTTPServer 8000"},
			VirtMode:        "host",
			Isolate:         "false",
			User:            "root",
			Group:           "root",
			MemoryGuarantee: "600Mb",
			MemoryLimit:     "2Gb",
			CpuGuarantee:    "0.5c",
			CpuLimit:        "3c",
			Controllers:     map[string]string{"devices": "false"},
		},
	}
	emptyCtx = &pb.Context{
		Vars: make([]*pb.Context_Var, 0),
	}
)

func TestFromYAMLs(t *testing.T) {
	type args struct {
		ctxBytes []byte
		objBytes []byte
		path     string
	}
	type want struct {
		ctx         *pb.Context
		meta        *pb.ObjectMeta
		packageSet  *pb.PackageSetSpec
		portoDaemon *pb.PortoDaemon
	}
	tests := []struct {
		name    string
		args    args
		want    want
		wantErr bool
	}{{
		name: "simple case package set",
		args: args{
			ctxBytes: []byte(ctxYAML),
			objBytes: []byte(packageSetYAML),
			path:     "/specs/PackageSet.yaml",
		},
		want: want{
			ctx:         ctxPb,
			meta:        packageSetMeta,
			packageSet:  packageSetSpec,
			portoDaemon: nil,
		},
		wantErr: false,
	}, {
		name: "simple case porto daemon",
		args: args{
			ctxBytes: []byte(ctxYAML),
			objBytes: []byte(portoDaemonYAML),
			path:     "/specs/PortoDaemon.yaml",
		},
		want: want{
			ctx:         ctxPb,
			meta:        portoDaemonMeta,
			packageSet:  nil,
			portoDaemon: portoDaemonSpec,
		},
		wantErr: false,
	}, {
		name: "invalid kind",
		args: args{
			ctxBytes: []byte(ctxYAML),
			objBytes: []byte(portoDaemonWithInvalidKindYAML),
			path:     "/specs/InvalidKind.yaml",
		},
		want: want{
			ctx:         nil,
			meta:        nil,
			packageSet:  nil,
			portoDaemon: nil,
		},
		wantErr: true,
	}, {
		name: "empty ctx package set",
		args: args{
			ctxBytes: nil,
			objBytes: []byte(packageSetYAML),
			path:     "/specs/PackageSet.yaml",
		},
		want: want{
			ctx:         emptyCtx,
			meta:        packageSetMeta,
			packageSet:  packageSetSpec,
			portoDaemon: nil,
		},
		wantErr: false,
	}, {
		name: "empty ctx porto daemon",
		args: args{
			ctxBytes: nil,
			objBytes: []byte(portoDaemonYAML),
			path:     "/specs/PortoDaemon.yaml",
		},
		want: want{
			ctx:         emptyCtx,
			meta:        portoDaemonMeta,
			packageSet:  nil,
			portoDaemon: portoDaemonSpec,
		},
		wantErr: false,
	}, {
		name: "empty ctx porto daemon with skip remove",
		args: args{
			ctxBytes: nil,
			objBytes: []byte(packageSetWithSkipRemoveYAML),
			path:     "/specs/PackageSet.yaml",
		},
		want: want{
			ctx:         emptyCtx,
			meta:        packageSetMetaWithSkipRemove,
			packageSet:  packageSetSpec,
			portoDaemon: nil,
		},
		wantErr: false,
	}, {
		name: "empty ctx invalid kind",
		args: args{
			ctxBytes: nil,
			objBytes: []byte(portoDaemonWithInvalidKindYAML),
			path:     "/specs/InvalidKind.yaml",
		},
		want: want{
			ctx:         nil,
			meta:        nil,
			packageSet:  nil,
			portoDaemon: nil,
		},
		wantErr: true,
	},
		{
			name: "invalid fields in spec",
			args: args{
				ctxBytes: nil,
				objBytes: []byte(portoDaemonYAMLWithMisspelledFields),
				path:     "/specs/InvalidFields.yaml",
			},
			want: want{
				ctx:         nil,
				meta:        nil,
				packageSet:  nil,
				portoDaemon: nil,
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := fromYAMLs(tt.args.ctxBytes, tt.args.objBytes, tt.args.path)
			if (err != nil) != tt.wantErr {
				t.Errorf("FromYAMLs() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if err != nil {
				return
			}
			if got.Kind() == "PackageSet" {
				assertpb.Equal(t, tt.want.packageSet, got.spec.(*pb.PackageSetSpec))
			}
			if got.Kind() == "PortoDaemon" {
				assertpb.Equal(t, tt.want.portoDaemon, got.spec.(*pb.PortoDaemon))
			}
			gotMetaWithoutCtx := &pb.ObjectMeta{
				Kind:            got.meta.Kind,
				Name:            got.meta.Name,
				Version:         got.meta.Version,
				DeleteRequested: got.meta.DeleteRequested,
				Labels:          got.meta.Labels,
				Annotations:     got.meta.Annotations,
			}
			assertpb.Equal(t, tt.want.meta, gotMetaWithoutCtx)
			assertpb.Equal(t, tt.want.ctx, got.meta.Ctx)
		})
	}
}

func TestUnitFromReader(t *testing.T) {
	unit := fmt.Sprintf("---\n%s\n---\n%s", ctxYAML, packageSetYAML)
	reader := strings.NewReader(unit)
	u, err := Parse(definitions.New(reader, "mock"))
	if err != nil {
		t.Errorf("FromReader failed: %s", err)
		return
	}
	if fName := u.meta.Annotations["filename"]; fName != "mock" {
		t.Errorf("FromReader expected path 'mock', got '%s'", fName)
		return
	}
	if stage := u.meta.Annotations["stage"]; stage != "<default>" {
		t.Errorf("FromReader expected stage '<default>', got '%s'", stage)
		return
	}
}

func TestSplitYAMLBigDocument(t *testing.T) {
	buf := make([]byte, 64*1024)
	for i := range buf {
		buf[i] = '\n'
	}
	copy(buf, ctxYAML)
	copy(buf[32*1024:], "---")
	copy(buf[32*1024+4:], packageSetYAML)
	ctx, spec, err := splitYAML(definitions.New(bytes.NewReader(buf), "test"))
	if err != nil {
		t.Errorf("error splitting yaml: %v", err)
	}
	if !strings.HasPrefix(string(ctx), ctxYAML) {
		t.Errorf("ctx content should be same as input")
	}
	if !strings.HasPrefix(string(spec), packageSetYAML) {
		t.Errorf("spec content should be same as input")
	}
}

func absent(m *pb.ObjectMeta) bool {
	return m.DeleteRequested || m.Annotations["stage"] == "absent"
}
