package executil

import (
	"bytes"
	"context"
	"fmt"
	"os"
	"os/exec"
	"strings"
	"time"
)

type ExecFn func(ctx context.Context, c *Cmd) error

type Cmd struct {
	Cmd    string
	Args   []string
	Env    []string
	stdout bytes.Buffer
	stderr bytes.Buffer
	execFn func(ctx context.Context, c *Cmd) error
}

func NewArgv(argv ...string) *Cmd {
	return &Cmd{
		Cmd:    argv[0],
		Args:   argv[1:],
		stdout: bytes.Buffer{},
		stderr: bytes.Buffer{},
		execFn: Execute,
	}
}

func NewCmd(cmd string, args ...string) *Cmd {
	return &Cmd{
		Cmd:    cmd,
		Args:   args,
		stdout: bytes.Buffer{},
		stderr: bytes.Buffer{},
	}
}

func (c *Cmd) AddArg(arg ...string) *Cmd {
	return &Cmd{
		Cmd:    c.Cmd,
		Args:   append(c.Args, arg...),
		Env:    c.Env,
		stdout: bytes.Buffer{},
		stderr: bytes.Buffer{},
		execFn: c.execFn,
	}
}

func (c *Cmd) WithExecutor(executor ExecFn) *Cmd {
	cmd := *c
	cmd.execFn = executor
	return &cmd
}

func (c *Cmd) SetExecutor(executor func(ctx context.Context, c *Cmd) error) {
	c.execFn = executor
}

func (c *Cmd) ExecCtx(ctx context.Context) error {
	return c.execFn(ctx, c)
}

func Execute(ctx context.Context, c *Cmd) error {
	devnull, err := os.Open(os.DevNull)
	if err != nil {
		return fmt.Errorf("failed to open /dev/null '%s': %w", c.String(), err)
	}
	cmd := exec.CommandContext(ctx, c.Cmd, c.Args...)
	cmd.Env = c.Env
	cmd.Dir = "/"
	cmd.Stdin = devnull
	cmd.Stdout = &c.stdout
	cmd.Stderr = &c.stderr
	cmd.SysProcAttr = cmdProcAttr
	err = cmd.Run()
	if err != nil {
		return fmt.Errorf("failed to execute '%s': %w", c.String(), err)
	}
	return nil
}

func (c *Cmd) String() string {
	return c.Cmd + " " + strings.Join(c.Args, " ")
}

func (c *Cmd) Stdout() []byte {
	return c.stdout.Bytes()
}

func (c *Cmd) Stderr() []byte {
	return c.stderr.Bytes()
}

func (c *Cmd) SetEnv(env []string) {
	c.Env = env
}

type Status struct {
	Ok      bool
	Message string
}

// Run executable in subprocess returning stdout, stderr and status.
//
// Uses small buffer and should no be used for large outputs or streaming.
func CheckOutput(cmd *Cmd, timeout time.Duration, env []string) ([]byte, []byte, *Status) {
	status := &Status{
		Ok:      true,
		Message: "",
	}
	cmd.SetEnv(env)
	ctx, cancel := context.WithTimeout(context.Background(), timeout)
	defer cancel()
	err := cmd.ExecCtx(ctx)
	if err != nil {
		status.Ok = false
		status.Message = err.Error()
	}
	return cmd.Stdout(), cmd.Stderr(), status
}

func TailOf(buf []byte, l int, prefix string) string {
	b := strings.Builder{}
	b.WriteString(prefix)
	cut := len(buf) - l + 3 // Ellipsis "..."
	if cut <= 0 {
		b.Write(buf)
		return b.String()
	}
	b.WriteString("...")
	b.Write(buf[cut:])
	return b.String()
}

// Creates new environ slice with provided keys overridden or added to set.
func Environ(vars []string) []string {
	// We use Go feature that only last key will be used,
	// so we can simply add duplicates to the end to override, e.g. PATH
	return append(os.Environ(), vars...)
}
