package fileutil

import (
	"errors"
	"fmt"
	"os"
	"path"
)

func AtomicWrite(filepath string, content []byte, chmod os.FileMode, makeDirs bool, chown bool, uid, gid int) error {
	if makeDirs {
		dirname := path.Dir(filepath)
		if dirname != "/" {
			err := os.MkdirAll(dirname, 0755)
			if err != nil && !errors.Is(err, os.ErrExist) {
				return fmt.Errorf("failed to create %s: %w", dirname, err)
			}
		}
	}
	tmp := filepath + ".tmp"
	f, err := os.OpenFile(tmp, os.O_RDWR|os.O_CREATE|os.O_TRUNC, chmod)
	if err != nil {
		return fmt.Errorf("failed to create '%s': %w", tmp, err)
	}
	defer f.Close()
	if chown {
		err = f.Chown(uid, gid)
		if err != nil {
			return fmt.Errorf("failed to chown '%s': %w", tmp, err)
		}
	}
	if _, err = f.Write(content); err != nil {
		return fmt.Errorf("failed to write to '%s': %w", tmp, err)
	}
	if err := f.Sync(); err != nil {
		return fmt.Errorf("failed to sync: %w", err)
	}
	if err := os.Rename(tmp, filepath); err != nil {
		return fmt.Errorf("failed to rename %s to '%s': %w", tmp, filepath, err)
	}
	return nil
}
