// Juggler client push client package
package juggler

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"
)

func MakeWalle(service, status, reason string) Event {
	// always empty err
	d, _ := json.Marshal(struct {
		TimeStamp int64  `json:"timestamp"`
		Reason    string `json:"reason"`
	}{
		TimeStamp: time.Now().Unix(),
		Reason:    reason,
	})
	return Event{
		Description: string(d),
		Service:     service,
		Status:      status,
		Tags:        make([]string, 0),
	}
}

func (e Event) WithTags(tags []string) Event {
	return Event{
		Description: e.Description,
		Service:     e.Service,
		Status:      e.Status,
		Tags:        tags,
		Host:        e.Host,
	}
}

func (e Event) WithHostname(hostname string) Event {
	return Event{
		Description: e.Description,
		Service:     e.Service,
		Status:      e.Status,
		Tags:        e.Tags,
		Host:        hostname,
	}
}

type Event struct {
	Description string   `json:"description"`
	Service     string   `json:"service"`
	Status      string   `json:"status"`
	Tags        []string `json:"tags"`
	Host        string   `json:"host"`
}

type Request struct {
	Source string  `json:"source"`
	Events []Event `json:"events"`
}

type Reply struct {
	Status  bool   `json:"success"`
	Message string `json:"message"`
	Events  []struct {
		Code  int    `json:"code"`
		Error string `json:"error"`
	}
	AcceptedEvents int `json:"accepted_events"`
}

const (
	DefaultLocalURL = "http://localhost:31579"
)

type Client struct {
	ServerURL string
	Client    *http.Client
	Header    map[string]string
}

func NewClientWithURL(serverURL string) (client *Client) {
	return &Client{
		ServerURL: serverURL,
		Client: &http.Client{
			Timeout: time.Duration(5 * time.Second),
		},
		Header: make(map[string]string),
	}
}

// NewClient create client which push events to local juggler service
func NewClient() (client *Client) {
	return NewClientWithURL(DefaultLocalURL)
}

// SetHeader add heades which later will be appended to events on push
func (c *Client) SetHeader(key string, value string) {
	c.Header[key] = value
}

func (c *Client) makeRequest(ctx context.Context, json bool, requestBody string) ([]byte, error) {
	req, err := http.NewRequest("POST", c.ServerURL+"/events", strings.NewReader(requestBody))
	if err != nil {
		return nil, err
	}

	for k, v := range c.Header {
		req.Header.Set(k, v)
	}
	if json {
		req.Header.Set("Content-Type", "application/json")
	}
	req = req.WithContext(ctx)
	resp, err := c.Client.Do(req)
	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()
	reply, err := ioutil.ReadAll(resp.Body)
	return reply, err
}

// PushEvents submit events to juggler service
func (c *Client) PushEvents(ctx context.Context, events Request) error {
	eventCount := len(events.Events)
	requestBody, err := json.Marshal(&events)
	if err != nil {
		return err
	}
	responseBody, err := c.makeRequest(ctx, true, string(requestBody))
	if err != nil {
		return err
	}
	var result Reply
	err = json.Unmarshal(responseBody, &result)
	if err != nil {
		return err
	}
	if !result.Status {
		return fmt.Errorf("bad push status: %s", result.Message)
	}
	if result.AcceptedEvents != eventCount {
		var badEvents []string
		for idx, data := range result.Events {
			if data.Error != "" {
				badEvents = append(badEvents, events.Events[idx].Service+":"+data.Error)
			}
		}
		return fmt.Errorf("some events were not accepted: %s", strings.Join(badEvents[:], ","))
	}
	return nil
}

// Examples:
//	c := juggler.NewClient()
//	c.SetHeader("User-Agent", "my-service-name/0.1")
//	ctx := context.Background()
//	r := juggler.Request{
//		Source: "my-service-name",
//		Events: []juggler.Event{
//			juggler.Event{
//				Description: "My event description",
//				Service:     "my-service-name",
//				Status:      "OK",
//			},
//		},
//	}
//	err := c.PushEvents(ctx, r)
