package reporting

import (
	"errors"
	"fmt"
	"os"
	"strings"

	"github.com/spf13/pflag"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

const (
	ctypeFlag         = "ctype"
	reportAddrFlag    = "report-addr"
	skipReportingFlag = "skip-reporting"
)

type ReportParams pb.ReportParams
type ttyType interface {
	Stat() (os.FileInfo, error)
}

var tty ttyType = os.Stdout

func New() *ReportParams {
	return &ReportParams{
		SkipReporting: false,
		ReporterAddrs: nil,
		Ctype:         "",
		YasmEnv:       pb.EnvType_UNDEFINED,
	}
}

func (p *ReportParams) MustRegisterFlags(flags *pflag.FlagSet) {
	flags.String(ctypeFlag, p.Ctype, "ctype")
	flags.StringSlice(reportAddrFlag, p.ReporterAddrs,
		"reporter servers addr (e.g host1:port1,host2:port2,...), implicitly sets skip-reporting=false if tty")
	flags.Bool(skipReportingFlag, false,
		"skip reporting to hm-report (default: true if tty, false otherwise)")
}

func (p *ReportParams) InferFromHostInfo(h *pb.HostInfo) error {
	const tagPrefix = "rtc.stage-"
	ctype := ""
	for _, tag := range h.WalleTags {
		if strings.HasPrefix(tag, tagPrefix) {
			if ctype != "" {
				return fmt.Errorf("cannot determine ctype from tags '%s': multiple rtc.stage tags found",
					strings.Join(h.WalleTags, ", "))
			}
			ctype = tag[len(tagPrefix):]
		}
	}
	if ctype == "" {
		return fmt.Errorf("cannot determine ctype from tags '%s': no rtc.stage tags found",
			strings.Join(h.WalleTags, ", "))
	}
	p.Ctype = ctype
	return nil
}

func (p *ReportParams) InferFromNoop(noop bool) {
	if noop {
		p.YasmEnv = pb.EnvType_NOOP
	} else {
		p.YasmEnv = pb.EnvType_REAL
	}
}

func (p *ReportParams) InferFromManageRequest(m *pb.ManageRequest) {
	switch m.ManageMode.(type) {
	case *pb.ManageRequest_Inline, *pb.ManageRequest_Targets:
		p.SkipReporting = true
	case *pb.ManageRequest_All:
		p.SkipReporting = false
	}
}

func (p *ReportParams) InferFromCmdline(flags *pflag.FlagSet) error {
	ra, err := flags.GetStringSlice(reportAddrFlag)
	if err != nil {
		return fmt.Errorf("cannot get %s flag: %w", reportAddrFlag, err)
	}
	if len(ra) > 0 {
		p.ReporterAddrs = ra
	}
	info, err := tty.Stat()
	if err != nil {
		return fmt.Errorf("cannot stat stdout: %w", err)
	}
	skip, err := flags.GetBool(skipReportingFlag)
	if err != nil {
		return fmt.Errorf("cannot get %s flag: %w", skipReportingFlag, err)
	}
	if info.Mode()&os.ModeCharDevice != 0 {
		if len(ra) > 0 || len(p.ReporterAddrs) > 0 {
			p.SkipReporting = skip
		} else {
			p.SkipReporting = true
		}
	} else {
		if skip {
			p.SkipReporting = true
		}
	}
	ctype, err := flags.GetString(ctypeFlag)
	if err != nil {
		return fmt.Errorf("cannot get %s flag: %w", ctypeFlag, err)
	}
	if ctype != "" {
		p.Ctype = ctype
	}
	return nil
}

func (p *ReportParams) Proto() *pb.ReportParams {
	return (*pb.ReportParams)(p)
}

func (p *ReportParams) Validate() []error {
	var problems []string
	if p.Ctype == "" {
		problems = append(problems, "ctype field cannot be empty")
	}
	if p.YasmEnv == pb.EnvType_UNDEFINED {
		problems = append(problems, "yasm env should be set")
	}
	if len(p.ReporterAddrs) == 0 && !p.SkipReporting {
		problems = append(problems, "no report addresses defined and skip-report is false")
	}
	if len(problems) > 0 {
		rv := make([]error, 0, len(problems))
		for _, p := range problems {
			rv = append(rv, errors.New(p))
		}
		return rv
	} else {
		return nil
	}
}
