package reporting

import (
	"fmt"
	"os"
	"strings"
	"testing"
	"time"

	"github.com/spf13/pflag"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func TestReportParams_InferFromNoop(t *testing.T) {
	rp := New()
	rp.InferFromNoop(false)
	if rp.YasmEnv != pb.EnvType_REAL {
		t.Errorf("yasm env should be: %d but got: %d", pb.EnvType_REAL, rp.YasmEnv)
	}
	rp.InferFromNoop(true)
	if rp.YasmEnv != pb.EnvType_NOOP {
		t.Errorf("yasm env should be: %d but got: %d", pb.EnvType_NOOP, rp.YasmEnv)
	}
}

func TestReportParams_InferFromHostInfo(t *testing.T) {
	rp := New()
	h := &pb.HostInfo{WalleTags: []string{"rtc"}}
	err := rp.InferFromHostInfo(h)
	if err == nil {
		t.Errorf("empty stage tag: %v", err)
	}
	h.WalleTags = append(h.WalleTags, "rtc.stage-prestable")
	err = rp.InferFromHostInfo(h)
	if err != nil || rp.Ctype != "prestable" {
		t.Errorf("error: %v, ctypes\"prestable\"!=\"%s\"", err, rp.Ctype)
	}
	rp.Ctype = ""
	h.WalleTags = append(h.WalleTags, "rtc.stage-experiment")
	err = rp.InferFromHostInfo(h)
	if err == nil {
		t.Error("there should be an error for multiple stage tags")
	}
}

func TestReportParams_Validate(t *testing.T) {
	rp := New()
	errs := rp.Validate()
	// All errors
	if len(errs) != 3 {
		t.Error("empty rp should not pass any validation")
	}
	rp.ReporterAddrs = append(rp.ReporterAddrs, "addr")
	// Yasm env not set
	rp.Ctype = "mock"
	errs = rp.Validate()
	if !strings.Contains(errs[0].Error(), "yasm") {
		t.Error("rp without yasm env validation should have error about yasm env")
	}
	// Ctype not set
	rp.Ctype = ""
	rp.YasmEnv = pb.EnvType_NOOP
	errs = rp.Validate()
	if !strings.Contains(errs[0].Error(), "ctype") {
		t.Error("rp without ctype validation should have error about ctype")
	}
	// Skip-reporting not set and report addresses empty
	rp.Ctype = "mock"
	rp.SkipReporting = false
	rp.ReporterAddrs = nil
	errs = rp.Validate()
	if !strings.Contains(errs[0].Error(), "skip-report") {
		t.Error("rp with empty report addrs should have skip-report set")
	}
	rp.SkipReporting = true
	errs = rp.Validate()
	if errs != nil {
		t.Error("rp with empty report addrs and skip-report set should pass validation")
	}
	// Validation successful
	rp.Ctype = "mock"
	errs = rp.Validate()
	if errs != nil {
		t.Error("valid rp should pass validation")
	}
}

func TestReportParams_InferFromManageRequest(t *testing.T) {
	r := &pb.ManageRequest{ManageMode: &pb.ManageRequest_Inline{}}
	rp := &ReportParams{}
	rp.InferFromManageRequest(r)
	if rp.SkipReporting != true {
		t.Errorf("skip-reporting should be true for inline request")
	}
	r.ManageMode = &pb.ManageRequest_Targets{}
	rp.InferFromManageRequest(r)
	if rp.SkipReporting != true {
		t.Errorf("skip-reporting should be true for targets request")
	}
	r.ManageMode = &pb.ManageRequest_All{}
	rp.InferFromManageRequest(r)
	if rp.SkipReporting != false {
		t.Errorf("skip-reporting should be false for manage all request")
	}
}

type ttyMock struct {
	isTTY bool
}

func (t *ttyMock) Name() string {
	return "mock"
}

func (t *ttyMock) Mode() os.FileMode {
	if t.isTTY {
		return os.ModeCharDevice
	} else {
		return 0
	}
}

func (t *ttyMock) ModTime() time.Time {
	return time.Now()
}

func (t *ttyMock) IsDir() bool {
	return false
}

func (t *ttyMock) Sys() interface{} {
	return nil
}

func (t *ttyMock) Stat() (os.FileInfo, error) {
	return t, nil
}

func (t *ttyMock) Size() int64 {
	return 0
}

func TestReportParams_InferFromCmdline(t *testing.T) {
	rp := New()
	rp.SkipReporting = false
	tty = &ttyMock{isTTY: true}
	flags := pflag.NewFlagSet("mock", pflag.ContinueOnError)
	rp.MustRegisterFlags(flags)
	_ = flags.Parse([]string{})
	_ = rp.InferFromCmdline(flags)
	if rp.SkipReporting != true {
		t.Errorf("skip-reporting should be true if running from tty")
	}

	flags = pflag.NewFlagSet("mock", pflag.ContinueOnError)
	rp.MustRegisterFlags(flags)
	_ = flags.Parse([]string{fmt.Sprintf("--%s", reportAddrFlag), "a,b,c"})
	rp.SkipReporting = true
	_ = rp.InferFromCmdline(flags)
	if rp.SkipReporting != false {
		t.Errorf("skip-reporting should be false if running from tty and --report-addrs used")
	}

	flags = pflag.NewFlagSet("mock", pflag.ContinueOnError)
	rp.MustRegisterFlags(flags)
	_ = flags.Parse([]string{fmt.Sprintf("--%s", reportAddrFlag), "a,b,c", fmt.Sprintf("--%s", skipReportingFlag)})
	rp.SkipReporting = false
	_ = rp.InferFromCmdline(flags)
	if rp.SkipReporting != true {
		t.Errorf("skip-reporting should be true if running from tty, --report-addrs and --skip-reporting used")
	}

	tty = &ttyMock{isTTY: false}

	flags = pflag.NewFlagSet("mock", pflag.ContinueOnError)
	rp.MustRegisterFlags(flags)
	_ = flags.Parse([]string{fmt.Sprintf("--%s", skipReportingFlag)})
	_ = rp.InferFromCmdline(flags)
	if rp.SkipReporting != true {
		t.Errorf("skip-reporting should be true if running without tty and using --skip-reporting")
	}

	flags = pflag.NewFlagSet("mock", pflag.ContinueOnError)
	rp.MustRegisterFlags(flags)
	rp.SkipReporting = false
	_ = flags.Parse([]string{})
	_ = rp.InferFromCmdline(flags)
	if rp.SkipReporting != false {
		t.Errorf("skip-reporting should be false if running without tty and flags")
	}

	rp.Ctype = "ctype"
	rp.ReporterAddrs = []string{"addr"}
	rp.SkipReporting = false
	flags = pflag.NewFlagSet("mock", pflag.ContinueOnError)
	rp.MustRegisterFlags(flags)
	_ = flags.Parse([]string{fmt.Sprintf("--%s", skipReportingFlag),
		fmt.Sprintf("--%s", ctypeFlag), "mut",
		fmt.Sprintf("--%s", reportAddrFlag), "a,b,c"})
	_ = rp.InferFromCmdline(flags)
	if rp.SkipReporting != true || len(rp.ReporterAddrs) != 3 || rp.Ctype != "mut" {
		t.Errorf("cmdline flags should override existing values")
	}
}
