package slot

import (
	unit2 "a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func NewSlot(slot *pb.Slot) *S {
	s := &S{slot: slot}
	NormalizeStatus(s.Status())
	return s
}

type S struct {
	slot        *pb.Slot
	lastCurrent *Rev
	changed     bool
}

func (s *S) ResetPending() {
	s.Status().SetNotPending()
}

func (s *S) Name() string {
	return s.slot.Name
}

func (s *S) Status() *Status {
	return (*Status)(s.slot.Status)
}

func (s *S) Revs() []*Rev {
	revs := make([]*Rev, len(s.slot.Revs))
	for i, r := range s.slot.Revs {
		revs[i] = NewRevision(r)
	}
	return revs
}

func (s *S) Current() *Rev {
	for _, r := range s.Revs() {
		if r.Proto().Target == pb.RevisionTarget_CURRENT {
			return r
		}
	}
	return nil
}

func (s *S) LastCurrent() *Rev {
	return s.lastCurrent
}

func (s *S) Changed() bool {
	return s.changed
}

func (s *S) Removed() []*Rev {
	removed := make([]*Rev, 0)
	for _, r := range s.Revs() {
		if r.Proto().Target == pb.RevisionTarget_REMOVED {
			removed = append(removed, r)
		}
	}
	return removed
}

func (s *S) HasChanges(u *unit2.Unit) bool {
	c := s.Current()
	if u.Absent() {
		// If slot become absent
		// check if we had current revision means
		// that we we changed slot with removing unit
		return c != nil
	}
	// If either did not have any revision
	// or new ID is not equal to current.
	return c == nil || u.ID() != c.ID()
}

func (s *S) ApplyUnit(u *unit2.Unit) {
	if u.Absent() {
		if c := s.Current(); c != nil {
			c.Proto().Target = pb.RevisionTarget_REMOVED
		}
		return
	}
	// Do nothing if we have no changes
	if !s.HasChanges(u) {
		return
	}
	s.changed = true
	c := s.Current()
	s.lastCurrent = c
	if c != nil {
		c.Proto().Target = pb.RevisionTarget_REMOVED
	}
	rev := RevisionFromUnit(u, u.RevisionMeta())
	s.SetRevs(append(s.Revs(), rev))
}

func (s *S) SetRevs(revs []*Rev) {
	revsProto := make([]*pb.Rev, len(revs))
	for i, rev := range revs {
		revsProto[i] = rev.Proto()
	}
	s.slot.Revs = revsProto
}

func (s *S) UpdateMeta(meta *pb.SlotMeta) {
	s.slot.Meta = meta
}

func (s *S) Meta() *Meta {
	return MetaFromProto(s.slot.Meta)
}

func (s *S) Proto() *pb.Slot {
	return s.slot
}

func (s *S) Kind() kind.Kind {
	// On unit removal we can had only REMOVED revisions
	if c := s.Current(); c != nil {
		return c.Kind()
	}
	if rem := s.Removed(); len(rem) > 0 {
		lastRemoved := rem[0]
		return lastRemoved.Kind()
	}
	// Fallback value when we gc all revisions
	return kind.Kind(s.slot.Meta.XKind)
}
