package slot

import (
	unit2 "a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"sort"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

// Define custom alias for SlotStatus to implement custom shorthand methods.
// Conversion from *pb.SlotStatus to *Status is implicit.
// Conversion *Status -> *pb.SlotStatus via (*Status).Proto() method.
type Status pb.SlotStatus

type Slot interface {
	Name() string
	Revs() []*Rev
	SetRevs([]*Rev)
	UpdateMeta(*pb.SlotMeta)
	Meta() *Meta
	Status() *Status
	Current() *Rev
	LastCurrent() *Rev
	Changed() bool
	Removed() []*Rev
	ResetPending()
	ApplyUnit(u *unit2.Unit)
	HasChanges(*unit2.Unit) bool
	Proto() *pb.Slot
	Kind() kind.Kind
}

// HostCTLSlots convert map with slots in proto slice compatible with HostctlState
func HostCTLSlots(slots map[string]Slot) []*pb.Slot {
	slotsPb := make([]*pb.Slot, 0, len(slots))
	// Sort keys to ensure stable result, when we move from
	// map to array.
	names := make([]string, 0, len(slots))
	for name := range slots {
		names = append(names, name)
	}
	sort.Strings(names)
	for _, name := range names {
		s := slots[name]
		slotsPb = append(slotsPb, s.Proto())
	}
	return slotsPb
}

func SlotsFromPb(state *pb.HostctlState) map[string]Slot {
	ss := make(map[string]Slot, 50)
	if state == nil {
		return make(map[string]Slot)
	}
	for _, s := range state.Slots {
		ss[s.Name] = NewSlot(s)
	}
	return ss
}

func New(name string) Slot {
	return NewSlot(&pb.Slot{
		Name:   name,
		Status: NewEmptyPbSlotStatus(),
		Revs:   make([]*pb.Rev, 0),
		Meta:   &pb.SlotMeta{},
	})
}

func NewEmptySlotStatus() *Status {
	return (*Status)(NewEmptyPbSlotStatus())
}

func NewEmptyPbSlotStatus() *pb.SlotStatus {
	return &pb.SlotStatus{
		Pending:    &pb.Condition{},
		Ready:      &pb.Condition{},
		Removed:    &pb.Condition{},
		Throttled:  &pb.Condition{},
		Changed:    &pb.Condition{},
		Conflicted: &pb.Condition{},
		Installed:  &pb.Condition{},
		Running:    &pb.Condition{},
	}
}

func NormalizeStatus(status *Status) {
	if status.Removed == nil {
		status.Removed = &pb.Condition{}
	}
	if status.Changed == nil {
		status.Changed = &pb.Condition{}
	}
	if status.Throttled == nil {
		status.Throttled = &pb.Condition{}
	}
	if status.Conflicted == nil {
		status.Conflicted = &pb.Condition{}
	}
	if status.Ready == nil {
		status.Ready = &pb.Condition{}
	}
	if status.Pending == nil {
		status.Pending = &pb.Condition{}
	}
	if status.Running == nil {
		status.Running = &pb.Condition{}
	}
	if status.Installed == nil {
		status.Installed = &pb.Condition{}
	}
}
