package systemd

import (
	"context"
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/systemd/persist"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func emptyCtx() (context.Context, context.CancelFunc) {
	return context.WithTimeout(context.Background(), DefaultTimeout)
}

func Test_noop_Reload(t *testing.T) {
	s := NewNoopSystemd(NewFake())
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Reload(ctx, u, revID); err != nil {
		t.Errorf("Reload() error = %v", err)
	}
}

func Test_noop_Restart(t *testing.T) {
	s := NewNoopSystemd(NewFake())
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Restart(ctx, u, revID); err != nil {
		t.Errorf("Restart() error = %v", err)
	}
}

func Test_noop_Start(t *testing.T) {
	s := NewNoopSystemd(NewFake())
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Start(ctx, u, revID); err != nil {
		t.Errorf("Start() error = %v", err)
	}
}

func Test_noop_Status(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()
	f.On("Status", mock.Anything, mock.Anything).Return(&UnitStatus{
		ActiveState:      ActiveStateActive,
		SubState:         SubStateRunning,
		UnitFileState:    UnitFileStateEnabled,
		NeedDaemonReload: true,
		Outdated:         false,
	}, nil)
	p.On("IsCurrent", mock.Anything, mock.Anything).Return(false)

	s := NewNoopSystemd(f)
	u := ServiceUnit("test")
	revID := "revID"
	status, err := s.Status(u, revID)
	if err != nil {
		t.Errorf("Status() error = %v", err)
	}

	f.AssertNumberOfCalls(t, "Status", 1)
	assert.False(t, status.Outdated)
	assert.True(t, status.NeedDaemonReload)
	assert.Equal(t, ActiveStateActive, status.ActiveState)
	assert.Equal(t, SubStateRunning, status.SubState)
	assert.Equal(t, UnitFileStateEnabled, status.UnitFileState)
}

func Test_noop_Stop(t *testing.T) {
	s := NewNoopSystemd(NewFake())
	u := ServiceUnit("test")
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Stop(ctx, u); err != nil {
		t.Errorf("Stop() error = %v", err)
	}
}

func Test_noop_Disable(t *testing.T) {
	s := NewNoopSystemd(NewFake())
	u := ServiceUnit("test")
	if err := s.Disable(u); err != nil {
		t.Errorf("Disable() error = %v", err)
	}
}

func Test_noop_Enable(t *testing.T) {
	s := NewNoopSystemd(NewFake())
	u := ServiceUnit("test")
	if err := s.Enable(u); err != nil {
		t.Errorf("Enable() error = %v", err)
	}
}

func Test_noop_ReloadDaemon(t *testing.T) {
	s := NewNoopSystemd(NewFake())
	if err := s.ReloadDaemon(); err != nil {
		t.Errorf("ReloadDaemon() error = %v", err)
	}
}
