package persist

import (
	"a.yandex-team.ru/infra/hostctl/internal/fileutil"
	"fmt"
	"io/ioutil"
	"os"
)

type Persist interface {
	IsCurrent(name, revID string) (bool, error)
	SaveRevision(name, revID string) error
	RemoveRevision(name string) error
}

func NewReal() Persist {
	return &persist{}
}

func NewNoop() Persist {
	return &noop{}
}

const (
	etagPath = "/var/lib/hostctl/private/"
	kind     = "system-service"
)

type persist struct {
}

func (*persist) SaveRevision(name, revID string) error {
	return fileutil.AtomicWrite(fmt.Sprintf("%s%s-%s.sha1", etagPath, kind, name), []byte(revID), 0644, true, false, 0, 0)
}

func (*persist) RemoveRevision(name string) error {
	return os.Remove(fmt.Sprintf("%s%s-%s.sha1", etagPath, kind, name))
}

func (*persist) IsCurrent(name, revID string) (bool, error) {
	f, err := os.Open(fmt.Sprintf("%s%s-%s.sha1", etagPath, kind, name))
	if err != nil {
		if os.IsNotExist(err) {
			return false, nil
		} else {
			return false, err
		}
	}
	defer f.Close()
	content, err := ioutil.ReadAll(f)
	if err != nil {
		return false, err
	}
	return string(content) == revID, nil
}

type noop struct {
}

func (*noop) SaveRevision(name, revID string) error {
	return nil
}

func (*noop) RemoveRevision(name string) error {
	return nil
}

func (n *noop) IsCurrent(name, revID string) (bool, error) {
	return true, nil
}
