package systemd

import (
	"context"
	"errors"
	"fmt"
	"os"
	"strconv"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/systemd/persist"
	"a.yandex-team.ru/infra/hostctl/internal/systemd/sysv"
	"github.com/coreos/go-systemd/v22/dbus"
	godbus "github.com/godbus/dbus/v5"
)

func NewSystemd(p persist.Persist) Systemd {
	return &systemd{p: p, sysv: sysv.NewReal()}
}

type systemd struct {
	p    persist.Persist
	sysv sysv.SysV
}

func (s *systemd) Restart(ctx context.Context, u *Unit, revID string) error {
	c, err := dbus.NewSystemdConnectionContext(ctx)
	if err != nil {
		return err
	}
	_, err = c.RestartUnitContext(ctx, u.FullName(), "replace", make(chan string))
	if err != nil {
		return fmt.Errorf("failed to restart systemd service: %w", err)
	}
	return s.p.SaveRevision(persistName(u), revID)
}

func (s *systemd) Reload(ctx context.Context, u *Unit, revID string) error {
	c, err := dbus.NewSystemdConnectionContext(ctx)
	if err != nil {
		return err
	}
	_, err = c.ReloadUnitContext(ctx, u.FullName(), "replace", make(chan string))
	if err != nil {
		return fmt.Errorf("failed to reload systemd service: %w", err)
	}
	return s.p.SaveRevision(persistName(u), revID)
}

func (s *systemd) Start(ctx context.Context, u *Unit, revID string) error {
	c, err := dbus.NewSystemdConnectionContext(ctx)
	if err != nil {
		return err
	}
	_, err = c.StartUnitContext(ctx, u.FullName(), "replace", make(chan string))
	if err != nil {
		return fmt.Errorf("failed to start systemd service: %w", err)
	}
	return s.p.SaveRevision(persistName(u), revID)
}

func (s *systemd) Stop(ctx context.Context, u *Unit) error {
	c, err := dbus.NewSystemdConnectionContext(ctx)
	if err != nil {
		return err
	}
	ch := make(chan string)
	_, err = c.StopUnitContext(ctx, u.FullName(), "fail", ch)
	if err != nil {
		return fmt.Errorf("failed to start systemd service: %w", err)
	}
	return s.p.RemoveRevision(persistName(u))
}

const (
	// On some overloaded hosts even simple calls can take a long time.
	// To avoid spurious timeouts - increase to some large but sane value.
	dbusTimeout = time.Minute
)

func (s *systemd) Enable(unit *Unit) error {
	ctx, cancel := context.WithTimeout(context.Background(), dbusTimeout)
	defer cancel()
	c, err := dbus.NewSystemdConnectionContext(ctx)
	if err != nil {
		return err
	}
	_, _, err = c.EnableUnitFilesContext(ctx, []string{unit.FullName()}, false, false)
	if err != nil {
		if isUnitNotFoundErr(err) {
			// this error can be caused by the fact that we managing SysV unit
			// for emulation /lib/systemd/systemd-sysv-install we had code below
			// https://github.com/systemd/systemd/blob/master/src/systemctl/systemctl-sysv-compat.c#L111
			isSysv, err2 := s.sysv.IsSysv(unit.Name)
			if err2 != nil {
				return fmt.Errorf("failed to determinate is SysV or not: '%s': '%w'", unit.Name, err2)
			}
			if isSysv {
				// fallback for sysv
				return s.sysv.Enable(ctx, unit.Name)
			}
		}
		return fmt.Errorf("failed to enable systemd service: %w", err)
	}
	return nil
}

func (s *systemd) Disable(unit *Unit) error {
	ctx, cancel := context.WithTimeout(context.Background(), dbusTimeout)
	defer cancel()
	c, err := dbus.NewSystemdConnectionContext(ctx)
	if err != nil {
		return err
	}
	_, err = c.DisableUnitFilesContext(ctx, []string{unit.FullName()}, false)
	if err != nil {
		if isUnitNotFoundErr(err) {
			// this error can be caused by the fact that we managing SysV unit
			// for emulation /lib/systemd/systemd-sysv-install we had code below
			// https://github.com/systemd/systemd/blob/master/src/systemctl/systemctl-sysv-compat.c#L111
			isSysv, err2 := s.sysv.IsSysv(unit.Name)
			if err2 != nil {
				return fmt.Errorf("failed to determinate is SysV or not: '%s': '%w'", unit.Name, err2)
			}
			if isSysv {
				// fallback for sysv
				return s.sysv.Disable(ctx, unit.Name)
			}
		}
		return fmt.Errorf("failed to disable systemd service: %w", err)
	}
	return nil
}

func (s *systemd) Status(unit *Unit, revID string) (*UnitStatus, error) {
	// increase timeout because we doing 2 dbus calls and 1 subproc call in this timeout
	ctx, cancel := context.WithTimeout(context.Background(), 2*dbusTimeout)
	defer cancel()
	c, err := dbus.NewSystemConnectionContext(ctx)
	if err != nil {
		return nil, err
	}
	props, err := c.GetAllPropertiesContext(ctx, unit.FullName())
	if err != nil {
		return nil, fmt.Errorf("failed to get properties systemd service: %w", err)
	}
	status, err := decodeProps(props)
	if err != nil {
		return nil, err
	}
	unitFileState, err := s.getUnitFileState(ctx, unit)
	if err != nil {
		return nil, err
	}
	status.UnitFileState = unitFileState
	// call IsCurrent in the end to calls all methods
	// with context with timeout before this long operation
	current, err := s.p.IsCurrent(persistName(unit), revID)
	if err != nil {
		return nil, err
	}
	status.Outdated = !current
	return status, nil
}

func (s *systemd) ReloadDaemon() error {
	ctx, cancel := context.WithTimeout(context.Background(), dbusTimeout)
	defer cancel()
	c, err := dbus.NewSystemdConnectionContext(ctx)
	if err != nil {
		return err
	}
	err = c.ReloadContext(ctx)
	if err != nil {
		return fmt.Errorf("failed to reload-daemon systemd service: %w", err)
	}
	return nil
}

// GetAllProperties returns not actual info about UnitFileState before daemon-reload called
// we do not want to reload daemon every time we update our status
// and we call GetUnitFileState dbus method to gets actual info about UnitFileState
// Also we do not have this method in go-systemd lib :(
func (s *systemd) getUnitFileState(ctx context.Context, u *Unit) (UnitFileState, error) {
	c, err := systemdConn(ctx)
	if err != nil {
		return UnitFileStateUnknown, fmt.Errorf("failed to open dbus conn: %w", err)
	}
	var unitFileState godbus.Variant
	err = c.
		Object("org.freedesktop.systemd1", "/org/freedesktop/systemd1").
		CallWithContext(ctx, "org.freedesktop.systemd1.Manager.GetUnitFileState", 0, u.FullName()).
		Store(&unitFileState)
	if err != nil {
		if isUnitNotFoundErr(err) {
			// this error can be caused by the fact that we managing SysV unit
			// for emulation /lib/systemd/systemd-sysv-install we had code below
			// https://github.com/systemd/systemd/blob/master/src/systemctl/systemctl-sysv-compat.c#L111
			isSysv, err2 := s.sysv.IsSysv(u.Name)
			if err2 != nil {
				return UnitFileStateUnknown, fmt.Errorf("failed to determinate is SysV or not: '%s': '%w'", u.Name, err2)
			}
			// fallback for SysV units
			if isSysv {
				if s.sysv.IsEnabled(ctx, u.Name) {
					return UnitFileStateEnabled, nil
				}
				return UnitFileStateDisabled, nil
			}
		}
		return UnitFileStateUnknown, err
	}
	state, ok := unitFileState.Value().(string)
	if !ok {
		return UnitFileStateUnknown, errors.New("failed to assert UnitFileState to string")
	}
	return UnitFileState(state), nil
}

func persistName(u *Unit) string {
	n := u.FullName()
	// for backward capability
	if u.Kind == Service {
		n = u.Name
	}
	return n
}

func systemdConn(ctx context.Context) (*godbus.Conn, error) {
	conn, err := godbus.Dial("unix:path=/run/systemd/private", godbus.WithContext(ctx))
	if err != nil {
		return nil, err
	}
	// Only use EXTERNAL method, and hardcode the uid (not username)
	// to avoid a username lookup (which requires a dynamically linked
	// libc)
	methods := []godbus.Auth{godbus.AuthExternal(strconv.Itoa(os.Getuid()))}
	err = conn.Auth(methods)
	if err != nil {
		_ = conn.Close()
		return nil, err
	}
	return conn, nil
}

func isUnitNotFoundErr(err error) bool {
	if err == nil {
		return false
	}
	// "No such file or directory" is error that we got from dbus methods calls
	return err.Error() == "No such file or directory"
}
