package systemd

import (
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/systemd/persist"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func Test_shadow_Reload(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()
	f.On("Reload", mock.Anything, mock.Anything).Return(nil)
	p.On("SaveRevision", mock.Anything, mock.Anything).Return(nil)

	s := NewShadowSystemctl(f, p)
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Reload(ctx, u, revID); err != nil {
		t.Errorf("Reload() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "SaveRevision", 1)
	p.AssertCalled(t, "SaveRevision", persistName(u), revID)
	f.AssertNumberOfCalls(t, "Reload", 0)
}

func Test_shadow_Restart(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()
	f.On("Restart", mock.Anything, mock.Anything).Return(nil)
	p.On("SaveRevision", mock.Anything, mock.Anything).Return(nil)

	s := NewShadowSystemctl(f, p)
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Restart(ctx, u, revID); err != nil {
		t.Errorf("Restart() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "SaveRevision", 1)
	p.AssertCalled(t, "SaveRevision", persistName(u), revID)
	f.AssertNumberOfCalls(t, "Restart", 0)
}

func Test_shadow_Start(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()
	f.On("Start", mock.Anything, mock.Anything).Return(nil)
	p.On("SaveRevision", mock.Anything, mock.Anything).Return(nil)

	s := NewShadowSystemctl(f, p)
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Start(ctx, u, revID); err != nil {
		t.Errorf("Start() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "SaveRevision", 1)
	p.AssertCalled(t, "SaveRevision", persistName(u), revID)
	f.AssertNumberOfCalls(t, "Start", 0)
}

func Test_shadow_Status(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()
	f.On("Status", mock.Anything, mock.Anything).Return(&UnitStatus{
		ActiveState:      ActiveStateActive,
		SubState:         SubStateRunning,
		UnitFileState:    UnitFileStateEnabled,
		NeedDaemonReload: true,
		Outdated:         false,
	}, nil)
	p.On("IsCurrent", mock.Anything, mock.Anything).Return(false)

	s := NewShadowSystemctl(f, p)
	u := ServiceUnit("test")
	revID := "revID"
	status, err := s.Status(u, revID)
	if err != nil {
		t.Errorf("Status() error = %v", err)
	}

	f.AssertNumberOfCalls(t, "Status", 1)
	assert.False(t, status.Outdated)
	assert.True(t, status.NeedDaemonReload)
	assert.Equal(t, ActiveStateActive, status.ActiveState)
	assert.Equal(t, SubStateRunning, status.SubState)
	assert.Equal(t, UnitFileStateEnabled, status.UnitFileState)
}

func Test_shadow_Stop(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()
	f.On("Stop", mock.Anything, mock.Anything).Return(nil)
	p.On("RemoveRevision", mock.Anything).Return(nil)

	s := NewShadowSystemctl(f, p)
	u := ServiceUnit("test")
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Stop(ctx, u); err != nil {
		t.Errorf("Stop() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "RemoveRevision", 1)
	p.AssertCalled(t, "RemoveRevision", persistName(u))
	f.AssertNumberOfCalls(t, "Stop", 0)
}

func Test_shadow_Disable(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()

	s := NewShadowSystemctl(f, p)
	u := ServiceUnit("test")
	if err := s.Disable(u); err != nil {
		t.Errorf("Disable() error = %v", err)
	}
}

func Test_shadow_Enable(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()

	s := NewShadowSystemctl(f, p)
	u := ServiceUnit("test")
	if err := s.Enable(u); err != nil {
		t.Errorf("Enable() error = %v", err)
	}
}

func Test_shadow_ReloadDaemon(t *testing.T) {
	f := NewFake()
	p := persist.NewFake()

	s := NewShadowSystemctl(f, p)
	if err := s.ReloadDaemon(); err != nil {
		t.Errorf("ReloadDaemon() error = %v", err)
	}
}
