package systemd

import (
	"reflect"
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/executil"
	"a.yandex-team.ru/infra/hostctl/internal/systemd/persist"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

var (
	ntpShowOutput = []byte("Type=simple\n" +
		"Restart=always\n" +
		"PIDFile=/var/run/ntp.pid\n" +
		"NotifyAccess=none\n" +
		"RestartUSec=100ms\n" +
		"TimeoutStartUSec=1min 30s\n" +
		"TimeoutStopUSec=1min 30s\n" +
		"RuntimeMaxUSec=infinity\n" +
		"WatchdogUSec=0\n" +
		"WatchdogTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"WatchdogTimestampMonotonic=3546324941543\n" +
		"FailureAction=none\n" +
		"PermissionsStartOnly=no\n" +
		"RootDirectoryStartOnly=no\n" +
		"RemainAfterExit=no\n" +
		"GuessMainPID=yes\n" +
		"MainPID=392895\n" +
		"ControlPID=0\n" +
		"FileDescriptorStoreMax=0\n" +
		"NFileDescriptorStore=0\n" +
		"StatusErrno=0\n" +
		"Result=success\n" +
		"ExecMainStartTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"ExecMainStartTimestampMonotonic=3546324941508\n" +
		"ExecMainExitTimestampMonotonic=0\n" +
		"ExecMainPID=392895\n" +
		"ExecMainCode=0\n" +
		"ExecMainStatus=0\n" +
		"ExecStart={ path=/usr/sbin/ntpd ; argv[]=/usr/sbin/ntpd -g -n -I eth0 -I eth1 -I eth2 -I eth3 -I eth4 -6 --pidfile /var/run/ntp.pid -u ntp:ntp ; ignore_errors=no ; start_time=[n/a] ; stop_time=[n/a] ; pid=0 ; code=(null) ; status=0/0 }\n" +
		"Slice=system.slice\n" +
		"ControlGroup=/system.slice/ntp.service\n" +
		"MemoryCurrent=741376\n" +
		"CPUUsageNSec=1080290400364\n" +
		"TasksCurrent=1\n" +
		"Delegate=no\n" +
		"CPUAccounting=no\n" +
		"CPUShares=18446744073709551615\n" +
		"StartupCPUShares=18446744073709551615\n" +
		"CPUQuotaPerSecUSec=infinity\n" +
		"BlockIOAccounting=no\n" +
		"BlockIOWeight=18446744073709551615\n" +
		"StartupBlockIOWeight=18446744073709551615\n" +
		"MemoryAccounting=no\n" +
		"MemoryLimit=18446744073709551615\n" +
		"DevicePolicy=auto\n" +
		"TasksAccounting=no\n" +
		"TasksMax=18446744073709551615\n" +
		"UMask=0022\n" +
		"LimitCPU=18446744073709551615\n" +
		"LimitCPUSoft=18446744073709551615\n" +
		"LimitFSIZE=18446744073709551615\n" +
		"LimitFSIZESoft=18446744073709551615\n" +
		"LimitDATA=18446744073709551615\n" +
		"LimitDATASoft=18446744073709551615\n" +
		"LimitSTACK=18446744073709551615\n" +
		"LimitSTACKSoft=8388608\n" +
		"LimitCORE=18446744073709551615\n" +
		"LimitCORESoft=0\n" +
		"LimitRSS=18446744073709551615\n" +
		"LimitRSSSoft=18446744073709551615\n" +
		"LimitNOFILE=4096\n" +
		"LimitNOFILESoft=1024\n" +
		"LimitAS=18446744073709551615\n" +
		"LimitASSoft=18446744073709551615\n" +
		"LimitNPROC=1031575\n" +
		"LimitNPROCSoft=1031575\n" +
		"LimitMEMLOCK=65536\n" +
		"LimitMEMLOCKSoft=65536\n" +
		"LimitLOCKS=18446744073709551615\n" +
		"LimitLOCKSSoft=18446744073709551615\n" +
		"LimitSIGPENDING=1031575\n" +
		"LimitSIGPENDINGSoft=1031575\n" +
		"LimitMSGQUEUE=819200\n" +
		"LimitMSGQUEUESoft=819200\n" +
		"LimitNICE=0\n" +
		"LimitNICESoft=0\n" +
		"LimitRTPRIO=0\n" +
		"LimitRTPRIOSoft=0\n" +
		"LimitRTTIME=18446744073709551615\n" +
		"LimitRTTIMESoft=18446744073709551615\n" +
		"OOMScoreAdjust=0\n" +
		"Nice=0\n" +
		"IOScheduling=0\n" +
		"CPUSchedulingPolicy=0\n" +
		"CPUSchedulingPriority=0\n" +
		"TimerSlackNSec=50000\n" +
		"CPUSchedulingResetOnFork=no\n" +
		"NonBlocking=no\n" +
		"StandardInput=null\n" +
		"StandardOutput=journal\n" +
		"StandardError=inherit\n" +
		"TTYReset=no\n" +
		"TTYVHangup=no\n" +
		"TTYVTDisallocate=no\n" +
		"SyslogPriority=30\n" +
		"SyslogLevelPrefix=yes\n" +
		"SyslogLevel=6\n" +
		"SyslogFacility=3\n" +
		"SecureBits=0\n" +
		"CapabilityBoundingSet=18446744073709551615\n" +
		"AmbientCapabilities=0\n" +
		"MountFlags=0\n" +
		"PrivateTmp=no\n" +
		"PrivateNetwork=no\n" +
		"PrivateDevices=no\n" +
		"ProtectHome=no\n" +
		"ProtectSystem=no\n" +
		"SameProcessGroup=no\n" +
		"UtmpMode=init\n" +
		"IgnoreSIGPIPE=yes\n" +
		"NoNewPrivileges=no\n" +
		"SystemCallErrorNumber=0\n" +
		"RuntimeDirectoryMode=0755\n" +
		"KillMode=control-group\n" +
		"KillSignal=15\n" +
		"SendSIGKILL=yes\n" +
		"SendSIGHUP=no\n" +
		"Id=ntp.service\n" +
		"Names=ntp.service\n" +
		"Requires=system.slice sysinit.target\n" +
		"Wants=time-sync.target\n" +
		"WantedBy=multi-user.target\n" +
		"Conflicts=shutdown.target\n" +
		"Before=multi-user.target time-sync.target shutdown.target\n" +
		"After=sysinit.target system.slice network.target basic.target syslog.target systemd-journald.socket\n" +
		"Description=Set time via NTP\n" +
		"LoadState=loaded\n" +
		"ActiveState=active\n" +
		"SubState=running\n" +
		"FragmentPath=/etc/systemd/system/ntp.service\n" +
		"UnitFileState=enabled\n" +
		"UnitFilePreset=enabled\n" +
		"StateChangeTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"StateChangeTimestampMonotonic=3546324941544\n" +
		"InactiveExitTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"InactiveExitTimestampMonotonic=3546324941544\n" +
		"ActiveEnterTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"ActiveEnterTimestampMonotonic=3546324941544\n" +
		"ActiveExitTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"ActiveExitTimestampMonotonic=3546324938570\n" +
		"InactiveEnterTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"InactiveEnterTimestampMonotonic=3546324939344\n" +
		"CanStart=yes\n" +
		"CanStop=yes\n" +
		"CanReload=no\n" +
		"CanIsolate=no\n" +
		"StopWhenUnneeded=no\n" +
		"RefuseManualStart=no\n" +
		"RefuseManualStop=no\n" +
		"AllowIsolate=no\n" +
		"DefaultDependencies=yes\n" +
		"OnFailureJobMode=replace\n" +
		"IgnoreOnIsolate=no\n" +
		"NeedDaemonReload=no\n" +
		"JobTimeoutUSec=infinity\n" +
		"JobTimeoutAction=none\n" +
		"ConditionResult=yes\n" +
		"AssertResult=yes\n" +
		"ConditionTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"ConditionTimestampMonotonic=3546324940440\n" +
		"AssertTimestamp=Wed 2020-10-07 13:48:16 MSK\n" +
		"AssertTimestampMonotonic=3546324940440\n" +
		"Transient=no\n" +
		"StartLimitInterval=10000000\n" +
		"StartLimitBurst=5\n" +
		"StartLimitAction=none\n")
	unknownService = []byte("Restart=no\n" +
		"NotifyAccess=none\n" +
		"RestartUSec=100ms\n" +
		"TimeoutStartUSec=1min 30s\n" +
		"TimeoutStopUSec=1min 30s\n" +
		"RuntimeMaxUSec=infinity\n" +
		"WatchdogUSec=0\n" +
		"WatchdogTimestampMonotonic=0\n" +
		"FailureAction=none\n" +
		"PermissionsStartOnly=no\n" +
		"RootDirectoryStartOnly=no\n" +
		"RemainAfterExit=no\n" +
		"GuessMainPID=yes\n" +
		"MainPID=0\n" +
		"ControlPID=0\n" +
		"FileDescriptorStoreMax=0\n" +
		"NFileDescriptorStore=0\n" +
		"StatusErrno=0\n" +
		"Result=success\n" +
		"ExecMainStartTimestampMonotonic=0\n" +
		"ExecMainExitTimestampMonotonic=0\n" +
		"ExecMainPID=0\n" +
		"ExecMainCode=0\n" +
		"ExecMainStatus=0\n" +
		"MemoryCurrent=18446744073709551615\n" +
		"CPUUsageNSec=18446744073709551615\n" +
		"TasksCurrent=18446744073709551615\n" +
		"Delegate=no\n" +
		"CPUAccounting=no\n" +
		"CPUShares=18446744073709551615\n" +
		"StartupCPUShares=18446744073709551615\n" +
		"CPUQuotaPerSecUSec=infinity\n" +
		"BlockIOAccounting=no\n" +
		"BlockIOWeight=18446744073709551615\n" +
		"StartupBlockIOWeight=18446744073709551615\n" +
		"MemoryAccounting=no\n" +
		"MemoryLimit=18446744073709551615\n" +
		"DevicePolicy=auto\n" +
		"TasksAccounting=no\n" +
		"TasksMax=18446744073709551615\n" +
		"UMask=0022\n" +
		"LimitCPU=18446744073709551615\n" +
		"LimitCPUSoft=18446744073709551615\n" +
		"LimitFSIZE=18446744073709551615\n" +
		"LimitFSIZESoft=18446744073709551615\n" +
		"LimitDATA=18446744073709551615\n" +
		"LimitDATASoft=18446744073709551615\n" +
		"LimitSTACK=18446744073709551615\n" +
		"LimitSTACKSoft=8388608\n" +
		"LimitCORE=18446744073709551615\n" +
		"LimitCORESoft=0\n" +
		"LimitRSS=18446744073709551615\n" +
		"LimitRSSSoft=18446744073709551615\n" +
		"LimitNOFILE=1048576\n" +
		"LimitNOFILESoft=1048576\n" +
		"LimitAS=18446744073709551615\n" +
		"LimitASSoft=18446744073709551615\n" +
		"LimitNPROC=1031575\n" +
		"LimitNPROCSoft=1031575\n" +
		"LimitMEMLOCK=65536\n" +
		"LimitMEMLOCKSoft=65536\n" +
		"LimitLOCKS=18446744073709551615\n" +
		"LimitLOCKSSoft=18446744073709551615\n" +
		"LimitSIGPENDING=1031575\n" +
		"LimitSIGPENDINGSoft=1031575\n" +
		"LimitMSGQUEUE=819200\n" +
		"LimitMSGQUEUESoft=819200\n" +
		"LimitNICE=0\n" +
		"LimitNICESoft=0\n" +
		"LimitRTPRIO=0\n" +
		"LimitRTPRIOSoft=0\n" +
		"LimitRTTIME=18446744073709551615\n" +
		"LimitRTTIMESoft=18446744073709551615\n" +
		"OOMScoreAdjust=0\n" +
		"Nice=0\n" +
		"IOScheduling=0\n" +
		"CPUSchedulingPolicy=0\n" +
		"CPUSchedulingPriority=0\n" +
		"TimerSlackNSec=50000\n" +
		"CPUSchedulingResetOnFork=no\n" +
		"NonBlocking=no\n" +
		"StandardInput=null\n" +
		"StandardOutput=inherit\n" +
		"StandardError=inherit\n" +
		"TTYReset=no\n" +
		"TTYVHangup=no\n" +
		"TTYVTDisallocate=no\n" +
		"SyslogPriority=30\n" +
		"SyslogLevelPrefix=yes\n" +
		"SyslogLevel=6\n" +
		"SyslogFacility=3\n" +
		"SecureBits=0\n" +
		"CapabilityBoundingSet=18446744073709551615\n" +
		"AmbientCapabilities=0\n" +
		"MountFlags=0\n" +
		"PrivateTmp=no\n" +
		"PrivateNetwork=no\n" +
		"PrivateDevices=no\n" +
		"ProtectHome=no\n" +
		"ProtectSystem=no\n" +
		"SameProcessGroup=no\n" +
		"UtmpMode=init\n" +
		"IgnoreSIGPIPE=yes\n" +
		"NoNewPrivileges=no\n" +
		"SystemCallErrorNumber=0\n" +
		"RuntimeDirectoryMode=0755\n" +
		"KillMode=control-group\n" +
		"KillSignal=15\n" +
		"SendSIGKILL=yes\n" +
		"SendSIGHUP=no\n" +
		"Id=wtf.service\n" +
		"Names=wtf.service\n" +
		"Description=wtf.service\n" +
		"LoadState=not-found\n" +
		"ActiveState=inactive\n" +
		"SubState=dead\n" +
		"StateChangeTimestampMonotonic=0\n" +
		"InactiveExitTimestampMonotonic=0\n" +
		"ActiveEnterTimestampMonotonic=0\n" +
		"ActiveExitTimestampMonotonic=0\n" +
		"InactiveEnterTimestampMonotonic=0\n" +
		"CanStart=no\n" +
		"CanStop=no\n" +
		"CanReload=no\n" +
		"CanIsolate=no\n" +
		"StopWhenUnneeded=no\n" +
		"RefuseManualStart=no\n" +
		"RefuseManualStop=no\n" +
		"AllowIsolate=no\n" +
		"DefaultDependencies=yes\n" +
		"OnFailureJobMode=replace\n" +
		"IgnoreOnIsolate=no\n" +
		"NeedDaemonReload=no\n" +
		"JobTimeoutUSec=infinity\n" +
		"JobTimeoutAction=none\n" +
		"ConditionResult=no\n" +
		"AssertResult=no\n" +
		"ConditionTimestampMonotonic=0\n" +
		"AssertTimestampMonotonic=0\n" +
		"LoadError=org.freedesktop.DBus.Error.FileNotFound \"No such file or directory\"\n" +
		"Transient=no\n" +
		"StartLimitInterval=10000000\n" +
		"StartLimitBurst=5\n" +
		"StartLimitAction=none\n")
)

func testSystemctl(p persist.Persist, cmds *cmds) *systemctl {
	l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
	return &systemctl{
		cmds: cmds,
		p:    p,
		l:    l,
	}
}

func testSystemctlCmd(method string, executor executil.ExecFn) *executil.Cmd {
	return executil.NewCmd(binSystemctl, "--system", method).WithExecutor(executor)
}

func Test_systemctl_Start(t *testing.T) {
	c := &cmds{
		Start: testSystemctlCmd("start", executil.FakeSystemctlOk),
	}
	p := persist.NewFake()
	p.On("SaveRevision", mock.Anything, mock.Anything).Return(nil)

	s := testSystemctl(p, c)
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Start(ctx, u, revID); err != nil {
		t.Errorf("Start() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "SaveRevision", 1)
	p.AssertCalled(t, "SaveRevision", persistName(u), revID)
}

func Test_systemctl_Disable(t *testing.T) {
	c := &cmds{
		Disable:      testSystemctlCmd("disable", executil.FakeSystemctlOk).AddArg("--no-reload"),
		ReloadDaemon: testSystemctlCmd("daemon-reload", executil.FakeSystemctlOk),
	}
	p := persist.NewFake()

	s := testSystemctl(p, c)
	u := ServiceUnit("test")
	if err := s.Disable(u); err != nil {
		t.Errorf("Disable() error = %v", err)
	}
}

func Test_systemctl_Enable(t *testing.T) {
	c := &cmds{
		Enable:       testSystemctlCmd("enable", executil.FakeSystemctlOk).AddArg("--no-reload"),
		ReloadDaemon: testSystemctlCmd("daemon-reload", executil.FakeSystemctlOk),
	}
	p := persist.NewFake()

	s := testSystemctl(p, c)
	u := ServiceUnit("test")
	if err := s.Enable(u); err != nil {
		t.Errorf("Enable() error = %v", err)
	}
}

func Test_systemctl_Reload(t *testing.T) {
	c := &cmds{
		Reload: testSystemctlCmd("reload", executil.FakeSystemctlOk),
	}
	p := persist.NewFake()
	p.On("SaveRevision", mock.Anything, mock.Anything).Return(nil)

	s := testSystemctl(p, c)
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Reload(ctx, u, revID); err != nil {
		t.Errorf("Reload() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "SaveRevision", 1)
	p.AssertCalled(t, "SaveRevision", persistName(u), revID)
}

func Test_systemctl_ReloadDaemon(t *testing.T) {
	c := &cmds{
		ReloadDaemon: testSystemctlCmd("daemon-reload", executil.FakeSystemctlOk),
	}
	p := persist.NewFake()

	s := testSystemctl(p, c)
	if err := s.ReloadDaemon(); err != nil {
		t.Errorf("ReloadDaemon() error = %v", err)
	}
}

func Test_systemctl_Restart(t *testing.T) {
	c := &cmds{
		Restart: testSystemctlCmd("restart", executil.FakeSystemctlOk),
	}
	p := persist.NewFake()
	p.On("SaveRevision", mock.Anything, mock.Anything).Return(nil)

	s := testSystemctl(p, c)
	u := ServiceUnit("test")
	revID := "revID"
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Restart(ctx, u, revID); err != nil {
		t.Errorf("Restart() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "SaveRevision", 1)
	p.AssertCalled(t, "SaveRevision", persistName(u), revID)
}

func Test_systemctl_Status(t *testing.T) {
	c := &cmds{
		Status:    testSystemctlCmd("status", executil.FakeSystemctlStatus(ntpShowOutput)),
		IsEnabled: testSystemctlCmd("is-enabled", executil.FakeSystemctlIsEnabled(string(UnitFileStateEnabledRuntime), true)),
	}
	p := persist.NewFake()
	p.On("IsCurrent", mock.Anything, mock.Anything).Return(true, nil)

	s := testSystemctl(p, c)
	u := ServiceUnit("test")
	revID := "revID"
	status, err := s.Status(u, revID)
	if err != nil {
		t.Errorf("Status() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "IsCurrent", 1)
	p.AssertCalled(t, "IsCurrent", persistName(u), revID)
	assert.Equal(t, false, status.Outdated)
	assert.Equal(t, false, status.NeedDaemonReload)
	assert.Equal(t, SubStateRunning, status.SubState)
	assert.Equal(t, ActiveStateActive, status.ActiveState)
	assert.Equal(t, UnitFileStateEnabledRuntime, status.UnitFileState)
}

func Test_systemctl_Stop(t *testing.T) {
	c := &cmds{
		Stop: testSystemctlCmd("stop", executil.FakeSystemctlOk),
	}
	p := persist.NewFake()
	p.On("RemoveRevision", mock.Anything).Return(nil)

	s := testSystemctl(p, c)
	u := ServiceUnit("test")
	ctx, cancel := emptyCtx()
	defer cancel()
	if err := s.Stop(ctx, u); err != nil {
		t.Errorf("Stop() error = %v", err)
	}

	p.AssertNumberOfCalls(t, "RemoveRevision", 1)
	p.AssertCalled(t, "RemoveRevision", persistName(u))
}

func Test_systemctl_unitFileState(t *testing.T) {
	p := persist.NewFake()
	tests := []struct {
		name              string
		isEnabledExecutor executil.ExecFn
		want              UnitFileState
		wantErr           bool
	}{{
		name:              string(UnitFileStateEnabled),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateEnabled), true),
		want:              UnitFileStateEnabled,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateEnabledRuntime),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateEnabledRuntime), true),
		want:              UnitFileStateEnabledRuntime,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateDisabled),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateDisabled), false),
		want:              UnitFileStateDisabled,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateLinked),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateLinked), false),
		want:              UnitFileStateLinked,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateLinkedRuntime),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateLinkedRuntime), false),
		want:              UnitFileStateLinkedRuntime,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateMasked),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateMasked), false),
		want:              UnitFileStateMasked,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateMaskedRuntime),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateMaskedRuntime), false),
		want:              UnitFileStateMaskedRuntime,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateStatic),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateStatic), true),
		want:              UnitFileStateStatic,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateIndirect),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateIndirect), true),
		want:              UnitFileStateIndirect,
		wantErr:           false,
	}, {
		name:              string(UnitFileStateBad),
		isEnabledExecutor: executil.FakeSystemctlIsEnabled(string(UnitFileStateBad), false),
		want:              UnitFileStateBad,
		wantErr:           true,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &cmds{
				IsEnabled: testSystemctlCmd("is-enabled", tt.isEnabledExecutor),
			}
			s := testSystemctl(p, c)
			got, err := s.unitFileState(ServiceUnit(""))
			if (err != nil) != tt.wantErr {
				t.Errorf("unitFileState() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("unitFileState() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_parseShowOutput(t *testing.T) {
	tests := []struct {
		name        string
		statusBytes []byte
		want        *UnitStatus
		wantErr     bool
	}{{
		name:        "ntp.service",
		statusBytes: ntpShowOutput,
		want: &UnitStatus{
			ActiveState:      ActiveStateActive,
			SubState:         SubStateRunning,
			NeedDaemonReload: false,
			Outdated:         false,
			Type:             TypeSimple,
		},
		wantErr: false,
	}, {
		name:        "unknown wtf.service",
		statusBytes: unknownService,
		want: &UnitStatus{
			ActiveState:      ActiveStateInactive,
			SubState:         SubStateDead,
			NeedDaemonReload: false,
		},
		wantErr: false,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := parseShowOutput(tt.statusBytes)
			if (err != nil) != tt.wantErr {
				t.Errorf("parseShowOutput() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("parseShowOutput() got = %v, want %v", got, tt.want)
			}
		})
	}
}
