package systemd

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/slices"
)

type Systemd interface {
	Enable(*Unit) error
	Disable(*Unit) error
	Restart(ctx context.Context, u *Unit, revID string) error
	Reload(ctx context.Context, u *Unit, revID string) error
	Start(ctx context.Context, u *Unit, revID string) error
	Stop(ctx context.Context, u *Unit) error
	Status(u *Unit, revID string) (*UnitStatus, error)
	ReloadDaemon() error
}

type UnitKind string

const (
	Service UnitKind = "service"
	Timer   UnitKind = "timer"
)

func ServiceUnit(name string) *Unit {
	return &Unit{Kind: Service, Name: name}
}

func TimerUnit(name string) *Unit {
	return &Unit{Kind: Timer, Name: name}
}

type Unit struct {
	Kind UnitKind
	Name string
}

func (u *Unit) FullName() string {
	return fmt.Sprintf("%s.%s", u.Name, u.Kind)
}

type UnitStatus struct {
	ActiveState      ActiveState
	SubState         SubState
	UnitFileState    UnitFileState
	LoadState        LoadState
	NeedDaemonReload bool
	Outdated         bool
	Type             Type
	ExecMain         *ExecMain
}

type ExecMain struct {
	Start          time.Time
	Stop           time.Time
	ExecMainStatus string
}

func (s *UnitStatus) Duration() time.Duration {
	if s.JobRunning() {
		return time.Since(s.ExecMain.Start)
	}
	return s.ExecMain.Stop.Sub(s.ExecMain.Start)
}

func (s *UnitStatus) Ok() bool {
	return s.ActiveState.Active() && !s.NeedDaemonReload && s.UnitFileState.Enabled() && !s.Outdated
}

func (s *UnitStatus) JobOk() bool {
	if s.NeedDaemonReload {
		return false
	}
	if !s.UnitFileState.Enabled() {
		return false
	}
	if s.Outdated {
		return false
	}
	return true
}

func (s *UnitStatus) JobRunning() bool {
	return s.ActiveState == ActiveStateActivating
}

func (s *UnitStatus) CopyFrom(n *UnitStatus) {
	s.ActiveState = n.ActiveState
	s.SubState = n.SubState
	s.UnitFileState = n.UnitFileState
	s.NeedDaemonReload = n.NeedDaemonReload
	s.Outdated = n.Outdated
	s.ExecMain = n.ExecMain
	s.Type = n.Type
	s.LoadState = n.LoadState
}

type SubState string

const (
	SubStateUnknown      SubState = "unknown"
	SubStateExited       SubState = "exited"
	SubStateDead         SubState = "dead"
	SubStateRunning      SubState = "running"
	SubStateFinalSigterm SubState = "final-sigterm"
	SubStateStartPre     SubState = "start-pre"
	SubStateStart        SubState = "start"
)

type Type string

const (
	TypeSimple  Type = "simple"
	TypeOneshot Type = "oneshot"
)

func (ss SubState) Running() bool {
	return ss == SubStateRunning || ss == SubStateFinalSigterm || ss == SubStateStartPre || ss == SubStateStart
}

type ActiveState string

const (
	ActiveStateUnknown      ActiveState = "unknown"
	ActiveStateActive       ActiveState = "active"
	ActiveStateActivating   ActiveState = "activating"
	ActiveStateReloading    ActiveState = "reloading"
	ActiveStateDeactivating ActiveState = "deactivating"
	ActiveStateInactive     ActiveState = "inactive"
)

func (as ActiveState) Active() bool {
	return as == ActiveStateActive || as == ActiveStateActivating || as == ActiveStateDeactivating || as == ActiveStateReloading
}

// ReloadedActive checks if unit became active after systemd service reload. Routine used in async polling during SystemService reload.
func (as ActiveState) ReloadedActive() bool {
	return as == ActiveStateActive || as == ActiveStateActivating || as == ActiveStateDeactivating
}

type LoadState string

const (
	LoadStateLoaded   LoadState = "loaded"
	LoadStateNotFound LoadState = "not-found"
)

func (ls LoadState) Loaded() bool {
	return ls == LoadStateLoaded
}

type UnitFileState string

const (
	UnitFileStateUnknown        UnitFileState = "unknown"
	UnitFileStateEnabled        UnitFileState = "enabled"
	UnitFileStateEnabledRuntime UnitFileState = "enabled-runtime"
	UnitFileStateDisabled       UnitFileState = "disabled"
	UnitFileStateStatic         UnitFileState = "static"
	UnitFileStateLinked         UnitFileState = "linked"
	UnitFileStateLinkedRuntime  UnitFileState = "linked-runtime"
	UnitFileStateMasked         UnitFileState = "masked"
	UnitFileStateMaskedRuntime  UnitFileState = "masked-runtime"
	UnitFileStateIndirect       UnitFileState = "indirect"
	UnitFileStateBad            UnitFileState = "bad"
)

var BadUnitFileStates = []string{
	string(UnitFileStateLinked),
	string(UnitFileStateLinkedRuntime),
	string(UnitFileStateMasked),
	string(UnitFileStateMaskedRuntime),
	string(UnitFileStateDisabled),
}

var EnabledUnitFileStates = []string{
	string(UnitFileStateEnabled),
	string(UnitFileStateStatic),
	string(UnitFileStateEnabledRuntime),
}

var DisabledUnitFileStates = []string{
	string(UnitFileStateDisabled),
	string(UnitFileStateStatic),
}

func (fs UnitFileState) Enabled() bool {
	return slices.ContainsString(EnabledUnitFileStates, string(fs))
}

func (fs UnitFileState) Disabled() bool {
	return slices.ContainsString(DisabledUnitFileStates, string(fs))
}

const (
	propType       = "Type"
	propActive     = "ActiveState"
	propSub        = "SubState"
	propNeedReload = "NeedDaemonReload"
	propFileState  = "UnitFileState"

	propExecMainStart  = "ExecMainStartTimestamp"
	propExecMainStop   = "ExecMainExitTimestamp"
	propExecMainStatus = "ExecMainStatus"
	propExecMainCode   = "ExecMainCode"
)

var requiredProps = []string{
	propActive,
	propSub,
	propNeedReload,
}

var optionalProps = []string{
	propType,
	propExecMainStart,
	propExecMainStop,
	propExecMainStatus,
}

var execMainProps = []string{
	propExecMainStart,
	propExecMainStop,
	propExecMainStatus,
}

// decodeProps decodes properties acquired from DBus into UnitStatus
func decodeProps(props map[string]interface{}) (status *UnitStatus, err error) {
	// Declare k before the loop to be able to refer in case of panic.
	var k string
	// Recover from failed type assertions.
	defer func() {
		if r := recover(); r != nil {
			err = fmt.Errorf("properties decoding failed at '%s': %s", k, r)
		}
	}()
	k = propActive
	v := props[k]
	t := v.(string)
	if len(t) == 0 {
		// Missing unit reported as having empty type.
		// At the moment we return default status instead of error.
		status = &UnitStatus{
			ActiveState:      ActiveStateInactive,
			SubState:         SubStateDead,
			NeedDaemonReload: false,
		}
		return
	}
	status = &UnitStatus{
		ActiveState: ActiveState(k),
	}
	for _, k := range requiredProps {
		v, ok := props[k]
		if !ok {
			err = fmt.Errorf("properties decoding failed: missing '%s'", k)
			return
		}
		switch k {
		case propActive:
			status.ActiveState = ActiveState(v.(string))
		case propSub:
			status.SubState = SubState(v.(string))
		case propNeedReload:
			status.NeedDaemonReload = v.(string) == "yes"
		}
	}
	v, ok := props[propType]
	// missing Type prop is possible on .timer units
	if !ok {
		return
	}
	status.Type = Type(v.(string))
	// we want to check execution time of oneshot jobs
	if status.Type == TypeOneshot {
		execMain, err := parseExecMain(props)
		if err != nil {
			return status, err
		}
		status.ExecMain = execMain
	}
	return
}

const timestampLayout = "Mon 2006-01-02 15:04:05 MST"

func parseExecMain(props map[string]interface{}) (execMain *ExecMain, err error) {
	var k string
	execMain = &ExecMain{}
	defer func() {
		if r := recover(); r != nil {
			err = fmt.Errorf("properties decoding failed at '%s': %s", k, r)
		}
	}()
	for _, k := range execMainProps {
		v, ok := props[k]
		if !ok {
			continue
		}
		switch k {
		case propExecMainStatus:
			// ExecMainStatus is exit-code of job process
			execMain.ExecMainStatus = v.(string)
		case propExecMainStart:
			execMain.Start, err = time.Parse(timestampLayout, v.(string))
		case propExecMainStop:
			execMain.Stop, err = time.Parse(timestampLayout, v.(string))
		}
		if err != nil {
			return nil, err
		}
	}
	return execMain, nil
}
