package systemd

import (
	"github.com/stretchr/testify/assert"
	"testing"
	"time"
)

// All possible property names acquired on random host via `systemctl show some-unit`
var propNames = []string{
	"Type",
	"UnitFileState",
	"Restart",
	"NotifyAccess",
	"RestartUSec",
	"TimeoutStartUSec",
	"TimeoutStopUSec",
	"TimeoutAbortUSec",
	"TimeoutStartFailureMode",
	"TimeoutStopFailureMode",
	"RuntimeMaxUSec",
	"WatchdogUSec",
	"WatchdogTimestampMonotonic",
	"RootDirectoryStartOnly",
	"RemainAfterExit",
	"GuessMainPID",
	"MainPID",
	"ControlPID",
	"FileDescriptorStoreMax",
	"NFileDescriptorStore",
	"StatusErrno",
	"Result",
	"ReloadResult",
	"CleanResult",
	"UID",
	"GID",
	"NRestarts",
	"ExecMainStartTimestampMonotonic",
	"ExecMainExitTimestampMonotonic",
	"ExecMainPID",
	"ExecMainCode",
	"ExecMainStatus",
	"MemoryCurrent",
	"CPUUsageNSec",
	"EffectiveCPUs",
	"EffectiveMemoryNodes",
	"TasksCurrent",
	"IPIngressBytes",
	"IPIngressPackets",
	"IPEgressBytes",
	"IPEgressPackets",
	"IOReadBytes",
	"IOReadOperations",
	"IOWriteBytes",
	"IOWriteOperations",
	"Delegate",
	"CPUAccounting",
	"CPUWeight",
	"StartupCPUWeight",
	"CPUShares",
	"StartupCPUShares",
	"CPUQuotaPerSecUSec",
	"CPUQuotaPeriodUSec",
	"AllowedCPUs",
	"AllowedMemoryNodes",
	"IOAccounting",
	"IOWeight",
	"StartupIOWeight",
	"BlockIOAccounting",
	"BlockIOWeight",
	"StartupBlockIOWeight",
	"MemoryAccounting",
	"DefaultMemoryLow",
	"DefaultMemoryMin",
	"MemoryMin",
	"MemoryLow",
	"MemoryHigh",
	"MemoryMax",
	"MemorySwapMax",
	"MemoryLimit",
	"DevicePolicy",
	"TasksAccounting",
	"TasksMax",
	"IPAccounting",
	"UMask",
	"LimitCPU",
	"LimitCPUSoft",
	"LimitFSIZE",
	"LimitFSIZESoft",
	"LimitDATA",
	"LimitDATASoft",
	"LimitSTACK",
	"LimitSTACKSoft",
	"LimitCORE",
	"LimitCORESoft",
	"LimitRSS",
	"LimitRSSSoft",
	"LimitNOFILE",
	"LimitNOFILESoft",
	"LimitAS",
	"LimitASSoft",
	"LimitNPROC",
	"LimitNPROCSoft",
	"LimitMEMLOCK",
	"LimitMEMLOCKSoft",
	"LimitLOCKS",
	"LimitLOCKSSoft",
	"LimitSIGPENDING",
	"LimitSIGPENDINGSoft",
	"LimitMSGQUEUE",
	"LimitMSGQUEUESoft",
	"LimitNICE",
	"LimitNICESoft",
	"LimitRTPRIO",
	"LimitRTPRIOSoft",
	"LimitRTTIME",
	"LimitRTTIMESoft",
	"RootHashSignature",
	"OOMScoreAdjust",
	"CoredumpFilter",
	"Nice",
	"IOSchedulingClass",
	"IOSchedulingPriority",
	"CPUSchedulingPolicy",
	"CPUSchedulingPriority",
	"CPUAffinity",
	"CPUAffinityFromNUMA",
	"NUMAPolicy",
	"NUMAMask",
	"TimerSlackNSec",
	"CPUSchedulingResetOnFork",
	"NonBlocking",
	"StandardInput",
	"StandardInputData",
	"StandardOutput",
	"StandardError",
	"TTYReset",
	"TTYVHangup",
	"TTYVTDisallocate",
	"SyslogPriority",
	"SyslogLevelPrefix",
	"SyslogLevel",
	"SyslogFacility",
	"LogLevelMax",
	"LogRateLimitIntervalUSec",
	"LogRateLimitBurst",
	"SecureBits",
	"CapabilityBoundingSet",
	"AmbientCapabilities",
	"DynamicUser",
	"RemoveIPC",
	"MountFlags",
	"PrivateTmp",
	"PrivateDevices",
	"ProtectClock",
	"ProtectKernelTunables",
	"ProtectKernelModules",
	"ProtectKernelLogs",
	"ProtectControlGroups",
	"PrivateNetwork",
	"PrivateUsers",
	"PrivateMounts",
	"ProtectHome",
	"ProtectSystem",
	"SameProcessGroup",
	"UtmpMode",
	"IgnoreSIGPIPE",
	"NoNewPrivileges",
	"SystemCallErrorNumber",
	"LockPersonality",
	"RuntimeDirectoryPreserve",
	"RuntimeDirectoryMode",
	"StateDirectoryMode",
	"CacheDirectoryMode",
	"LogsDirectoryMode",
	"ConfigurationDirectoryMode",
	"TimeoutCleanUSec",
	"MemoryDenyWriteExecute",
	"RestrictRealtime",
	"RestrictSUIDSGID",
	"RestrictNamespaces",
	"MountAPIVFS",
	"KeyringMode",
	"ProtectHostname",
	"KillMode",
	"KillSignal",
	"RestartKillSignal",
	"FinalKillSignal",
	"SendSIGKILL",
	"SendSIGHUP",
	"WatchdogSignal",
	"Id",
	"Names",
	"Description",
	"LoadState",
	"ActiveState",
	"FreezerState",
	"SubState",
	"StateChangeTimestampMonotonic",
	"InactiveExitTimestampMonotonic",
	"ActiveEnterTimestampMonotonic",
	"ActiveExitTimestampMonotonic",
	"InactiveEnterTimestampMonotonic",
	"CanStart",
	"CanStop",
	"CanReload",
	"CanIsolate",
	"CanFreeze",
	"StopWhenUnneeded",
	"RefuseManualStart",
	"RefuseManualStop",
	"AllowIsolate",
	"DefaultDependencies",
	"OnFailureJobMode",
	"IgnoreOnIsolate",
	"NeedDaemonReload",
	"JobTimeoutUSec",
	"JobRunningTimeoutUSec",
	"JobTimeoutAction",
	"ConditionResult",
	"AssertResult",
	"ConditionTimestampMonotonic",
	"AssertTimestampMonotonic",
	"LoadError",
	"Transient",
	"Perpetual",
	"StartLimitIntervalUSec",
	"StartLimitBurst",
	"StartLimitAction",
	"FailureAction",
	"SuccessAction",
	"CollectMode",
}

func TestDecodePropsNotFound(t *testing.T) {
	m := make(map[string]interface{})
	m[propActive] = ""
	s, err := decodeProps(m)
	if err != nil {
		t.Fatal(err)
	}
	assert.Equal(t, SubStateDead, s.SubState)
	assert.Equal(t, ActiveStateInactive, s.ActiveState)
	assert.Equal(t, false, s.NeedDaemonReload)
}

func TestDecodeProps(t *testing.T) {
	m := make(map[string]interface{})
	for _, k := range propNames {
		m[k] = "some"
	}
	m[propNeedReload] = "yes"
	m[propActive] = string(ActiveStateActivating)
	m[propSub] = string(SubStateDead)
	s, err := decodeProps(m)
	if err != nil {
		t.Fatal(err)
	}
	assert.Equal(t, SubStateDead, s.SubState)
	assert.Equal(t, ActiveStateActivating, s.ActiveState)
	assert.Equal(t, true, s.NeedDaemonReload)
}

func TestDecodeExecMainProps(t *testing.T) {
	m := make(map[string]interface{})
	for _, k := range propNames {
		m[k] = "some"
	}
	m[propExecMainStatus] = "0"
	e, err := parseExecMain(m)
	if err != nil {
		t.Fatal(err)
	}
	assert.Equal(t, "0", e.ExecMainStatus)
}

func TestDecodeExecMainPropsDuration(t *testing.T) {
	m := make(map[string]interface{})
	for _, k := range propNames {
		m[k] = "some"
	}
	m[propExecMainStatus] = "0"
	m[propExecMainStart] = "Thu 2021-01-14 09:55:57 MSK"
	m[propExecMainStop] = "Thu 2021-01-14 09:56:57 MSK"
	e, err := parseExecMain(m)
	if err != nil {
		t.Fatal(err)
	}
	assert.Equal(t, "0", e.ExecMainStatus)
	const timestampLayout = "Mon 2006-01-02 15:04:05 MST"
	start, _ := time.Parse(timestampLayout, "Thu 2021-01-14 09:55:57 MSK")
	stop, _ := time.Parse(timestampLayout, "Thu 2021-01-14 09:56:57 MSK")
	assert.Equal(t, start, e.Start)
	assert.Equal(t, stop, e.Stop)
}

func TestDecodePropsPanic(t *testing.T) {
	m := make(map[string]interface{})
	m[propActive] = false // Unexpected actual type, need string
	_, err := decodeProps(m)
	assert.EqualError(t, err,
		"properties decoding failed at 'ActiveState': interface conversion: interface {} is bool, not string")
}

func BenchmarkDecodeProps(b *testing.B) {
	// Generate a map of props
	m := make(map[string]interface{})
	for _, k := range propNames {
		m[k] = "some"
	}
	m[propNeedReload] = false
	b.ResetTimer()
	b.ReportAllocs()
	for i := 0; i < b.N; i++ {
		_, err := decodeProps(m)
		if err != nil {
			b.Fatal(err)
		}
	}
}
