package sysv

import (
	"a.yandex-team.ru/infra/hostctl/internal/executil"
	"context"
	"fmt"
	"os"
)

// SysV emulates /lib/systemd/systemd-sysv-install work
// https://github.com/systemd/systemd/blob/master/src/systemctl/systemd-sysv-install.SKELETON
type SysV interface {
	Enable(ctx context.Context, name string) error
	Disable(ctx context.Context, name string) error
	IsEnabled(ctx context.Context, name string) bool
	IsSysv(name string) (bool, error)
}

func NewReal() SysV {
	return &real{}
}

type real struct {
}

func (r *real) Enable(ctx context.Context, name string) error {
	cmd := executil.
		NewCmd("/lib/systemd/systemd-sysv-install", "enable", name).
		WithExecutor(executil.Execute)
	return cmd.ExecCtx(ctx)
}

func (r *real) Disable(ctx context.Context, name string) error {
	cmd := executil.
		NewCmd("/lib/systemd/systemd-sysv-install", "disable", name).
		WithExecutor(executil.Execute)
	return cmd.ExecCtx(ctx)
}

func (r *real) IsEnabled(ctx context.Context, name string) bool {
	cmd := executil.
		NewCmd("/lib/systemd/systemd-sysv-install", "is-enabled", name).
		WithExecutor(executil.Execute)
	return cmd.ExecCtx(ctx) == nil
}

// IsSysv checks if systemd unit is SysV unit
// stat /etc/init.d/<name> and if exist it is SysV unit
func (r *real) IsSysv(name string) (bool, error) {
	if _, err := os.Stat(fmt.Sprintf("/etc/init.d/%s", name)); err != nil {
		if os.IsNotExist(err) {
			return false, nil
		}
		return false, err
	}
	return true, nil
}
