package template

import (
	"a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"google.golang.org/protobuf/proto"
)

// MaterializedCtx defines an object holding ready to be used info for templates substitution
type MaterializedCtx interface {
	// Values returns evaluated values map to be used for substitution in templates
	Values() map[string]string
}

// CompiledCtx defines an object holding pre-compiled context ready for materialization with concrete HostInfo
type CompiledCtx interface {
	// Materialize evaluates context with provided HostInfo
	Materialize(hi *pb.HostInfo) (MaterializedCtx, error)
}

// CtxDefinition holds raw context parsed from YAML ctx document
type CtxDefinition pb.Context

// Proto casts CtxDefinition as underlying protobuf struct
func (d *CtxDefinition) Proto() *pb.Context {
	return (*pb.Context)(d)
}

// CtxParserFunc defines function used to parse ctx part of document
type CtxParserFunc func(document unitstorage.Document) (*CtxDefinition, error)

// CtxCompiler defines interface for compiling CtxDefinition into CompiledCtx
type CtxCompiler interface {
	// Compile compiles CtxDefinition into instance of CompiledCtx interface
	Compile(ctxDef *CtxDefinition) (CompiledCtx, error)
}

// MaterializedMeta represents pb.ObjectMeta instance after variable substitution
type MaterializedMeta pb.ObjectMeta

// MetaTemplate defines interface for object metadata before variables substitutions
type MetaTemplate interface {
	// Materialize substitutes variables in template using provided MaterializedCtx
	Materialize(ctx MaterializedCtx) (*MaterializedMeta, error)
}

// MaterializedSpec defines spec after variables substitutions
type MaterializedSpec proto.Message

// SpecTemplate defines interface for spec before variables substitutions
type SpecTemplate interface {
	// Materialize substitutes variables in template using provided MaterializedCtx
	Materialize(ctx MaterializedCtx) (MaterializedSpec, error)
}

// MetaParserFunc defines function interface used for parsing document to MetaTemplate
type MetaParserFunc func(document unitstorage.Document) (MetaTemplate, error)

// SpecParser defines interface for spec parser
type SpecParser interface {
	// ParseSpec parses `spec` field from spec part of Document to SpecTemplate
	ParseSpec(document unitstorage.Document, meta *MaterializedMeta) (SpecTemplate, error)
}

type Template interface {
	Render(hi *pb.HostInfo) (*unit.Unit, error)
	Path() string
	Name() string
}
