package template

import (
	"bytes"
	"errors"
	"fmt"
	"io"

	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
)

type document struct {
	ctxBuf  []byte
	specBuf []byte
	repo    string
	path    string
}

func (s *document) CtxReader() unitstorage.CtxReader {
	if s.ctxBuf != nil {
		return bytes.NewReader(s.ctxBuf)
	} else {
		return nil
	}
}

func (s *document) SpecReader() unitstorage.SpecReader {
	return bytes.NewReader(s.specBuf)
}

func (s *document) Path() string {
	return s.path
}

func (s *document) Repo() string {
	return s.repo
}

func (s *document) String() string {
	return fmt.Sprintf("repo: %s, path: %s", s.repo, s.path)
}

func NewDocument(ctxBuf, specBuf []byte, repo, path string) unitstorage.Document {
	return &document{ctxBuf: ctxBuf, specBuf: specBuf, repo: repo, path: path}
}

// DocumentFromStorage loads document from specified storage, splitting file into ctx and spec yaml documents
func DocumentFromStorage(s unitstorage.Storage, path string) (unitstorage.Document, error) {
	f, err := s.OpenFile(path)
	if err != nil {
		return nil, err
	}
	defer f.Reader.Close()
	return DocumentFromReader(f.Reader, f.Path, f.Repo)
}

// DocumentFromReader loads document from specified reader, splitting file into ctx and spec yaml documents
func DocumentFromReader(r io.Reader, path, repo string) (unitstorage.Document, error) {
	ctxBytes, specBytes, err := splitYAML(r)
	if err != nil {
		return nil, err
	}
	return NewDocument(
		ctxBytes,
		specBytes,
		repo,
		path,
	), nil
}

// splitYAML splits document from io.Reader and returns buffers with ctx and spec documents or error
func splitYAML(f io.Reader) (ctx []byte, spec []byte, err error) {
	s := yamlutil.NewDocumentDecoder(f).Scanner()
	documents := make([][]byte, 0)
	for s.Scan() {
		// Scanner overwrites underlying buffer, so we need a copy instead of slice
		doc := make([]byte, len(s.Bytes()))
		copy(doc, s.Bytes())
		documents = append(documents, doc)
	}
	if len(documents) <= 0 {
		return nil, nil, errors.New("no YAML documents")
	}
	if len(documents) > 2 {
		return nil, nil, fmt.Errorf("too many YAML documents: %d, "+
			"need 1 (obj) or 2 (ctx + obj)", len(documents))
	}
	if len(documents) == 2 {
		return documents[0], documents[1], nil
	}
	return nil, documents[0], nil
}
