package template

import (
	"fmt"
	"io"
	"strings"
	"testing"

	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	"github.com/stretchr/testify/assert"
)

var (
	ctx = `vars:
  - name: v1
    match:
      - exp: "default()"
        val: default`
	spec = `meta:
  name: mock
  kind: mock
spec:
  mock: mock`
)

func TestSplitYAML_CtxPresent(t *testing.T) {
	doc := fmt.Sprintf("---\n%s\n---\n%s", ctx, spec)
	reader := strings.NewReader(doc)
	ctxBytes, specBytes, err := splitYAML(reader)
	assert.NoError(t, err)
	assert.Equal(t, ctx, string(ctxBytes[4:])) // drop yaml document beginning
	assert.Equal(t, spec, string(specBytes))

	doc = fmt.Sprintf("%s\n---\n%s", ctx, spec)
	reader = strings.NewReader(doc)
	ctxBytes, specBytes, err = splitYAML(reader)
	assert.NoError(t, err)
	assert.Equal(t, ctx, string(ctxBytes))
	assert.Equal(t, spec, string(specBytes))
}

func TestSplitYAML_CtxAbsent(t *testing.T) {
	doc := spec
	reader := strings.NewReader(doc)
	ctxBytes, specBytes, err := splitYAML(reader)
	assert.NoError(t, err)
	assert.Nil(t, ctxBytes)
	assert.Equal(t, spec, string(specBytes))
}

func TestSplitYAML_NoDocuments(t *testing.T) {
	doc := ""
	reader := strings.NewReader(doc)
	ctxBytes, specBytes, err := splitYAML(reader)
	assert.EqualError(t, err, "no YAML documents")
	assert.Nil(t, ctxBytes)
	assert.Nil(t, specBytes)
}

func TestSplitYAML_ManyDocuments(t *testing.T) {
	doc := fmt.Sprintf("---\n%s\n---\n%s\n---\n%s\n---\n%s\n", ctx, spec, ctx, spec)
	reader := strings.NewReader(doc)
	ctxBytes, specBytes, err := splitYAML(reader)
	assert.EqualError(t, err, "too many YAML documents: 4, need 1 (obj) or 2 (ctx + obj)")
	assert.Nil(t, ctxBytes)
	assert.Nil(t, specBytes)
}

func TestDocumentFromStorage(t *testing.T) {
	name := "unit"
	path := "<mock>/unit.yaml"
	repo := "mock"
	ctx := "vars: []"
	spec := "meta:\n  name: unit"
	content := fmt.Sprintf("%s\n---\n%s", ctx, spec)
	storage := unitstorage.NewSingleUnitStorage(name, path, repo, content)
	doc, err := DocumentFromStorage(storage, name)
	assert.NoError(t, err)
	assert.Equal(t, path, doc.Path())
	assert.Equal(t, repo, doc.Repo())
	ctxBytes, err := io.ReadAll(doc.CtxReader())
	assert.NoError(t, err)
	assert.Equal(t, ctx, string(ctxBytes))
	specReader := doc.SpecReader()
	specBytes1, err := io.ReadAll(specReader)
	assert.NoError(t, err)
	_, err = specReader.Seek(0, io.SeekStart)
	assert.NoError(t, err)
	specBytes2, err := io.ReadAll(specReader)
	assert.NoError(t, err)
	assert.Equal(t, specBytes1, specBytes2)
	assert.Equal(t, spec, string(specBytes1))
}

func TestDocumentFromReader(t *testing.T) {
	path := "<mock>/unit.yaml"
	repo := "mock"
	ctx := "vars: []"
	spec := "meta:\n  name: unit"
	content := fmt.Sprintf("%s\n---\n%s", ctx, spec)
	doc, err := DocumentFromReader(strings.NewReader(content), path, repo)
	assert.NoError(t, err)
	assert.Equal(t, path, doc.Path())
	assert.Equal(t, repo, doc.Repo())
	ctxBytes, err := io.ReadAll(doc.CtxReader())
	assert.NoError(t, err)
	assert.Equal(t, ctx, string(ctxBytes))
	specReader := doc.SpecReader()
	specBytes1, err := io.ReadAll(specReader)
	assert.NoError(t, err)
	_, err = specReader.Seek(0, io.SeekStart)
	assert.NoError(t, err)
	specBytes2, err := io.ReadAll(specReader)
	assert.NoError(t, err)
	assert.Equal(t, specBytes1, specBytes2)
	assert.Equal(t, spec, string(specBytes1))
}
