package template

import (
	"errors"
	"fmt"
	"io"
	"regexp"

	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"github.com/golang/protobuf/proto"
	"google.golang.org/protobuf/encoding/protojson"
)

var fragmentListRE = regexp.MustCompile(`[0-9a-z-_][0-9a-z-_ ]+[0-9a-z-_]$`)

// ParseMetaDocument parses spec part of provided Document
// and extracts info from `meta` field to MetaTemplate
func ParseMetaDocument(document unitstorage.Document) (MetaTemplate, error) {
	rv := &pb.ObjectMeta{}
	specReader := document.SpecReader()
	if specReader == nil {
		return nil, fmt.Errorf("got nil specReader for document %s", document.String())
	}
	_, err := specReader.Seek(0, io.SeekStart)
	if err != nil {
		return nil, err
	}
	buf, err := io.ReadAll(specReader)
	if err != nil {
		return nil, err
	}
	err = unmarshalMeta(buf, rv)
	if err != nil {
		return nil, err
	}
	rv.Annotations = inferAnnotations(rv.Annotations, document)
	// Just validation. Fall on unsuccessful result
	_, err = parseSkipRemovePhase(rv)
	if err != nil {
		return nil, err
	}
	return NewMetaPbTemplate(rv), nil
}

// MetaPbTemplate aliases ObjectMeta to implement MetaTemplate interface
type MetaPbTemplate pb.ObjectMeta

// NewMetaPbTemplate casts ObjectMeta to MetaPbTemplate
func NewMetaPbTemplate(m *pb.ObjectMeta) *MetaPbTemplate {
	return (*MetaPbTemplate)(m)
}

func (t *MetaPbTemplate) Materialize(ctx MaterializedCtx) (*MaterializedMeta, error) {
	rv := proto.Clone((*pb.ObjectMeta)(t)).(*pb.ObjectMeta)
	err := doMeta(rv, ctx.Values())
	if err != nil {
		return nil, err
	}
	err = validateFragments((*MaterializedMeta)(rv))
	if err != nil {
		return nil, err
	}
	return (*MaterializedMeta)(rv), err
}

// inferAnnotations add automatic annotations
func inferAnnotations(annotations map[string]string, document unitstorage.Document) map[string]string {
	if annotations == nil {
		annotations = make(map[string]string)
	}
	annotations["filename"] = document.Path()
	annotations["repo"] = document.Repo()
	if len(annotations["stage"]) == 0 {
		// Fill in stage manually, was not required at some point.
		// For backward compatibility.
		annotations["stage"] = "<default>"
	}
	return annotations
}

// parseSkipRemovePhase parses value of special skip-remove-phase annotation
func parseSkipRemovePhase(meta *pb.ObjectMeta) (*pb.SkipRemovePhase, error) {
	f := &pb.SkipRemovePhase{}
	if forget := meta.Annotations["skip-remove-phase"]; len(forget) != 0 {
		err := yamlutil.UnmarshalStrict([]byte(forget), f)
		if err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(meta.annotation.forget): %w", err)
		}
	}
	return f, nil
}

// doMeta substitutes ObjectMeta fields using provided variables map
func doMeta(meta *pb.ObjectMeta, ctxMap map[string]string) error {
	// Substitute annotations only (not whole meta)
	if err := fmtMap(meta.Annotations, ctxMap); err != nil {
		return fmt.Errorf("failed to apply ctx to meta.annotations: %w", err)
	}
	// Format version
	v, err := fmtString(meta.Version, ctxMap)
	if err != nil {
		return err
	}
	meta.Version = v
	// Format kind
	k, err := fmtString(meta.Kind, ctxMap)
	if err != nil {
		return err
	}
	meta.Kind = k
	return nil
}

// validateFragments validates fragment usage
func validateFragments(meta *MaterializedMeta) error {
	if fStr, ok := meta.Annotations["fragments"]; ok && len(fStr) > 0 {
		if string(kind.HostPod) != meta.Kind {
			return errors.New("fragments allowed only in HostPod")
		}
		if !fragmentListRE.Match([]byte(fStr)) {
			return fmt.Errorf("meta.annotations.fragments filed should match re: %s", fragmentListRE.String())
		}
	}
	return nil
}

// unmarshalMeta unmarshalls `meta` field from spec buf
func unmarshalMeta(buf []byte, meta *pb.ObjectMeta) error {
	m := &struct {
		Meta *metaValue `yaml:"meta"`
	}{
		Meta: (*metaValue)(meta),
	}
	if err := yamlutil.UnmarshalStrict(buf, m); err != nil {
		return fmt.Errorf("can not unmarshall YAML(meta): %w", err)
	}
	return nil
}

// metaValue defines alias for custom UnmarshalJSON implementation
// using protojson instead of encoding/json
type metaValue pb.ObjectMeta

func (m *metaValue) UnmarshalJSON(b []byte) error {
	return protojson.Unmarshal(b, (*pb.ObjectMeta)(m))
}
