package template

import (
	"fmt"
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/test/assertpb"
	"github.com/stretchr/testify/assert"
)

func TestUnmarshalMeta(t *testing.T) {
	m := &pb.ObjectMeta{}
	buf := []byte("{malformed")
	err := unmarshalMeta(buf, m)
	assert.Error(t, err)
	buf = []byte(`meta:
  name: mock
  kind: kindMock
  version: verMock
`)
	err = unmarshalMeta(buf, m)
	assert.NoError(t, err)
	assert.Equal(t, "mock", m.Name)
	assert.Equal(t, "kindMock", m.Kind)
	assert.Equal(t, "verMock", m.Version)
}

func TestDoMeta(t *testing.T) {
	m := &pb.ObjectMeta{
		Version:     "{ver}",
		Annotations: map[string]string{"stage": "{stage}"},
	}
	ctxMap := map[string]string{"ver": "verMock", "stage": "stageMock"}
	err := doMeta(m, ctxMap)
	assert.NoError(t, err)
	assert.Equal(t, "verMock", m.Version)
	assert.Equal(t, "stageMock", m.Annotations["stage"])
}

func TestParseMetaDocument(t *testing.T) {
	specBuf := []byte(`meta:
  name: mock
  version: vmock
`)
	doc := NewDocument(nil, specBuf, "mock", "/mock.yaml")
	mt, err := ParseMetaDocument(doc)
	assert.NoError(t, err)
	expected := &MetaPbTemplate{
		Name:    "mock",
		Version: "vmock",
		Annotations: map[string]string{
			"filename": "/mock.yaml",
			"stage":    "<default>",
			"repo":     "mock",
		},
	}
	assertpb.Equal(t, (*pb.ObjectMeta)(expected), (*pb.ObjectMeta)(mt.(*MetaPbTemplate)))

	specBuf = []byte(`meta:
  name: mock
  version: vmock
  annotations:
    stage: smock
    filename: overrideme
`)
	doc = NewDocument(nil, specBuf, "mock", "/mock.yaml")
	mt, err = ParseMetaDocument(doc)
	assert.NoError(t, err)
	expected = &MetaPbTemplate{
		Name:    "mock",
		Version: "vmock",
		Annotations: map[string]string{
			"filename": "/mock.yaml",
			"stage":    "smock",
			"repo":     "mock",
		},
	}
	assertpb.Equal(t, (*pb.ObjectMeta)(expected), (*pb.ObjectMeta)(mt.(*MetaPbTemplate)))

	specBuf = []byte(`meta:
  name: mock
  version: vmock
  annotations:
    skip-remove-phase: |+
      {malformed
`)
	doc = NewDocument(nil, specBuf, "mock", "/mock.yaml")
	_, err = ParseMetaDocument(doc)
	assert.Error(t, err)
}

func TestMetaPbTemplate_Materialize(t *testing.T) {
	original := &MetaPbTemplate{
		Name:    "mock",
		Version: "{ver}",
		Annotations: map[string]string{
			"stage": "{stage}",
			"a1":    "{var1}",
		},
	}
	mt := &MetaPbTemplate{
		Name:    "mock",
		Version: "{ver}",
		Annotations: map[string]string{
			"stage": "{stage}",
			"a1":    "{var1}",
		},
	}
	ctx := NewMaterializedCtx(map[string]string{"ver": "vmock", "stage": "smock", "var1": "val1"})
	mm, err := mt.Materialize(ctx)
	assert.NoError(t, err)
	assertpb.Equal(t, (*pb.ObjectMeta)(original), (*pb.ObjectMeta)(mt))
	assert.Equal(t, "vmock", mm.Version)
	assert.Equal(t, "smock", mm.Annotations["stage"])
	assert.Equal(t, "val1", mm.Annotations["a1"])
}

func TestMetaPbTemplate_MaterializeWithFragments(t *testing.T) {
	original := &MetaPbTemplate{
		Kind:    string(kind.HostPod),
		Name:    "mock",
		Version: "{ver}",
		Annotations: map[string]string{
			"stage":     "{stage}",
			"a1":        "{var1}",
			"fragments": "{fragments}",
		},
	}
	mt := &MetaPbTemplate{
		Kind:    string(kind.HostPod),
		Name:    "mock",
		Version: "{ver}",
		Annotations: map[string]string{
			"stage":     "{stage}",
			"a1":        "{var1}",
			"fragments": "{fragments}",
		},
	}
	ctx := NewMaterializedCtx(map[string]string{"ver": "vmock", "stage": "smock", "var1": "val1", "fragments": "a b"})
	mm, err := mt.Materialize(ctx)
	assert.NoError(t, err)
	assertpb.Equal(t, (*pb.ObjectMeta)(original), (*pb.ObjectMeta)(mt))
	assert.Equal(t, "vmock", mm.Version)
	assert.Equal(t, "smock", mm.Annotations["stage"])
	assert.Equal(t, "val1", mm.Annotations["a1"])
	assert.Equal(t, "a b", mm.Annotations["fragments"])
}

func TestMetaPbTemplate_MaterializeWithKindTemplate(t *testing.T) {
	original := &MetaPbTemplate{
		Kind:    "{ukind}",
		Name:    "mock",
		Version: "{ver}",
		Annotations: map[string]string{
			"stage": "{stage}",
			"a1":    "{var1}",
		},
	}
	mt := &MetaPbTemplate{
		Kind:    "{ukind}",
		Name:    "mock",
		Version: "{ver}",
		Annotations: map[string]string{
			"stage": "{stage}",
			"a1":    "{var1}",
		},
	}
	ctx := NewMaterializedCtx(map[string]string{"ver": "vmock", "stage": "smock", "var1": "val1", "ukind": "HostPod"})
	mm, err := mt.Materialize(ctx)
	assert.NoError(t, err)
	assertpb.Equal(t, (*pb.ObjectMeta)(original), (*pb.ObjectMeta)(mt))
	assert.Equal(t, "HostPod", mm.Kind)
	assert.Equal(t, "vmock", mm.Version)
	assert.Equal(t, "smock", mm.Annotations["stage"])
	assert.Equal(t, "val1", mm.Annotations["a1"])
}

func TestInferAnnotations(t *testing.T) {
	doc := NewDocument(nil, nil, "mock", "/mock.yaml")
	r := inferAnnotations(nil, doc)
	assert.Equal(t, map[string]string{"filename": "/mock.yaml", "repo": "mock", "stage": "<default>"}, r)
	r = inferAnnotations(map[string]string{"stage": "mock", "filename": "overridden", "repo": "overridden"}, doc)
	assert.Equal(t, map[string]string{"filename": "/mock.yaml", "repo": "mock", "stage": "mock"}, r)
}

func TestParseSkipRemovePhase(t *testing.T) {
	m := &pb.ObjectMeta{Annotations: map[string]string{"skip-remove-phase": "{malformed"}}
	_, err := parseSkipRemovePhase(m)
	assert.Error(t, err)
	m.Annotations["skip-remove-phase"] = `daemon: true
packages:
- p1
- p2
files:
- f1
- f2`
	r, err := parseSkipRemovePhase(m)
	assert.NoError(t, err)
	expected := &pb.SkipRemovePhase{
		Packages: []string{"p1", "p2"},
		Files:    []string{"f1", "f2"},
		Daemon:   true,
	}
	assertpb.Equal(t, expected, r)
}

func TestValidateFragments(t *testing.T) {
	meta := &MaterializedMeta{
		Kind:    string(kind.HostPod),
		Name:    "yandex-iss-agent",
		Version: "4.0.0.0",
		Annotations: map[string]string{
			"fragments": "a b c",
		},
	}
	assert.NoError(t, validateFragments(meta))

	for _, k := range []string{
		string(kind.PackageSet),
		string(kind.PortoDaemon),
		string(kind.SystemService),
		string(kind.TimerJob)} {
		meta.Kind = k
		assert.Errorf(t, validateFragments(meta), "fragments allowed only in HostPod")
	}

	meta.Kind = string(kind.HostPod)
	meta.Annotations["fragments"] = " a b c "
	assert.Errorf(t, validateFragments(meta), fmt.Sprintf("meta.annotations.fragments filed should match re: %s", fragmentListRE.String()))
}
