package template

import (
	"fmt"
	"io"

	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/proto"
)

var jsonUnmarshalOpts = &protojson.UnmarshalOptions{DiscardUnknown: true}

// SpecPbTemplate holds spec template after parsing
type SpecPbTemplate struct {
	m proto.Message
}

// NewSpecPbTemplate creates instance of SpecPbTemplate
func NewSpecPbTemplate(m proto.Message) *SpecPbTemplate {
	return &SpecPbTemplate{m: m}
}

func (t *SpecPbTemplate) Materialize(ctx MaterializedCtx) (MaterializedSpec, error) {
	rv := proto.Clone(t.m)
	err := interpolatePb(ctx.Values(), rv)
	if err != nil {
		return nil, err
	}
	normalizeSpec(rv)
	return rv, nil
}

// DefaultSpecParser implements SpecParser interface
type DefaultSpecParser struct {
}

// NewDefaultSpecParser create new DefaultSpecParser instance
func NewDefaultSpecParser() *DefaultSpecParser {
	return &DefaultSpecParser{}
}

// ParseSpec parses spec from `spec` field of spec part of Document using provided MaterializedMeta
// spec type to be produced defined in MaterializedMeta
func (p *DefaultSpecParser) ParseSpec(document unitstorage.Document, meta *MaterializedMeta) (SpecTemplate, error) {
	specReader := document.SpecReader()
	if specReader == nil {
		return nil, fmt.Errorf("got nil specReader for document %s", document.String())
	}
	_, err := specReader.Seek(0, io.SeekStart)
	if err != nil {
		return nil, err
	}
	buf, err := io.ReadAll(specReader)
	if err != nil {
		return nil, err
	}
	switch kind.Kind(meta.Kind) {
	case kind.PackageSet:
		specPb := &pb.PackageSetSpec{}
		s := &struct {
			Spec *specPackageSetValue `yaml:"spec"`
		}{Spec: (*specPackageSetValue)(specPb)}
		if err := yamlutil.UnmarshalStrict(buf, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return NewSpecPbTemplate(specPb), nil
	case kind.PortoDaemon:
		specPb := &pb.PortoDaemon{}
		s := &struct {
			Spec *specPortoDaemonValue `yaml:"spec"`
		}{Spec: (*specPortoDaemonValue)(specPb)}
		if err := yamlutil.UnmarshalStrict(buf, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return NewSpecPbTemplate(specPb), nil
	case kind.SystemService:
		specPb := &pb.SystemServiceSpec{}
		s := &struct {
			Spec *specSystemServiceValue `yaml:"spec"`
		}{Spec: (*specSystemServiceValue)(specPb)}
		if err := yamlutil.UnmarshalStrict(buf, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return NewSpecPbTemplate(specPb), nil
	case kind.TimerJob:
		specPb := &pb.TimerJobSpec{}
		s := &struct {
			Spec *specTimerJobValue `yaml:"spec"`
		}{Spec: (*specTimerJobValue)(specPb)}
		if err := yamlutil.UnmarshalStrict(buf, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return NewSpecPbTemplate(specPb), nil
	case kind.HostPod, kind.HostPodFragment:
		specPb := &pb.HostPodSpec{}
		s := &struct {
			Spec *specPodValue `yaml:"spec"`
		}{Spec: (*specPodValue)(specPb)}
		if err := yamlutil.UnmarshalStrict(buf, s); err != nil {
			return nil, fmt.Errorf("can not unmarshall YAML(spec): %w", err)
		}
		return NewSpecPbTemplate(specPb), nil
	}
	return nil, fmt.Errorf("invalid unit kind: %s, possible kinds: [%s]", meta.Kind, kind.Kinds)
}

// specPackageSetValue defines alias for custom UnmarshalJSON implementation
// using protojson instead of encoding/json
type specPackageSetValue pb.PackageSetSpec

func (m *specPackageSetValue) UnmarshalJSON(b []byte) error {
	return jsonUnmarshalOpts.Unmarshal(b, (*pb.PackageSetSpec)(m))
}

// specSystemServiceValue defines alias for custom UnmarshalJSON implementation
// using protojson instead of encoding/json
type specSystemServiceValue pb.SystemServiceSpec

func (m *specSystemServiceValue) UnmarshalJSON(b []byte) error {
	return jsonUnmarshalOpts.Unmarshal(b, (*pb.SystemServiceSpec)(m))
}

// specTimerJobValue defines alias for custom UnmarshalJSON implementation
// using protojson instead of encoding/json
type specTimerJobValue pb.TimerJobSpec

func (m *specTimerJobValue) UnmarshalJSON(b []byte) error {
	return jsonUnmarshalOpts.Unmarshal(b, (*pb.TimerJobSpec)(m))
}

// specPortoDaemonValue defines alias for custom UnmarshalJSON implementation
// using protojson instead of encoding/json
type specPortoDaemonValue pb.PortoDaemon

func (m *specPortoDaemonValue) UnmarshalJSON(b []byte) error {
	return jsonUnmarshalOpts.Unmarshal(b, (*pb.PortoDaemon)(m))
}

// specPodValue defines alias for custom UnmarshalJSON implementation
// using protojson instead of encoding/json
type specPodValue pb.HostPodSpec

func (m *specPodValue) UnmarshalJSON(b []byte) error {
	return jsonUnmarshalOpts.Unmarshal(b, (*pb.HostPodSpec)(m))
}
